# coding: utf-8



import logging
import re

import cached_property
import cachetools.func

from django.conf import settings

from intranet.dogma.dogma.core.models import User
from intranet.dogma.dogma.core.dao.users import create_or_update_user_from_data
from ids.registry import registry
from ids.exceptions import IDSException


EMAIL_GUESSER_CACHE_TTL = 12 * 60 * 60  # 12h

staff_repository = registry.get_repository('staff',
                                           'person',
                                           user_agent='dogma',
                                           oauth_token=settings.DOGMA_OAUTH_TOKEN,
                                           retries=3,
                                           )

log = logging.getLogger(__name__)


def is_trusted_user(yauser):
    """
    Пользователь может просматривать все репозитории, даже закрытые.
    """
    return yauser.is_authenticated() and str(yauser.uid) in settings.TRUSTED_USERS


class EmailGuesser(object):

    @cached_property.threaded_cached_property_with_ttl(ttl=EMAIL_GUESSER_CACHE_TTL)
    def all_users(self):
        return list(User.objects.all())

    @cached_property.threaded_cached_property_with_ttl(ttl=60*60)
    def _user_mails_map(self):
        return {user.email.lower(): user for user in self.all_users}

    @cached_property.threaded_cached_property_with_ttl(ttl=60*60)
    def _user_other_mails_map(self):
        res = {}
        for user in self.all_users:
            for mail in user.other_emails_parsed:
                res[mail.lower()] = user
        return res

    def guess_user_by(self, user_email, message=None):
        user_email = user_email.lower()
        user, is_trusted_result = self._guess_user_by(user_email)
        return self._jupyter_hack(user, is_trusted_result, message)

    def _guess_user_by(self, user_email):
        user = self._user_mails_map.get(user_email)
        if user:
            return user, True

        if not settings.IS_BUSINESS:
            user = self._user_other_mails_map.get(user_email)
            if user:
                return user, True

        if not settings.IS_BUSINESS:
            try:
                user = self._get_user_from_staff(user_email)
            except UnicodeError:
                log.warning('Got UnicodeError while getting data "%s"', repr(user_email))
        return user, user is not None

    def _jupyter_hack(self, user, is_trusted_result, message=None):
        if not settings.IS_BUSINESS and is_trusted_result\
            and user.login == 'robot-jupyter-cloud' and message is not None:
            groups = re.search('Shared by (.+?)@ via https://jupyter.yandex-team.ru', message)
            if groups:
                return self.guess_user_by(groups.group(1) + '@yandex-team.ru')

        return user, is_trusted_result

    @cachetools.func.ttl_cache(maxsize=1 << 14, ttl=EMAIL_GUESSER_CACHE_TTL)
    def _get_user_from_staff(self, email):
        return get_user_from_staff_by_email(email)


def get_user_from_staff_by_email(email):
    user_data = get_user_data_by_email_from_staff(email=email)
    if user_data is not None:
        return create_or_update_user_from_data(user_data)


def get_user_data_by_email_from_staff(email):
    if not isinstance(email, str):
        email = email.decode('utf-8')
    try:
        user_data = staff_repository.get(lookup={'_query': settings.STAFF_REGEX.format(email),
                                                 '_fields': ','.join(settings.STAFF_FIELDS),
                                                 '_limit': '1',
                                                 }
                                         )
    except IDSException:
        log.exception('Got exception from staff-api with email: "%s"', email)

    else:
        if len(user_data) == 1:
            return user_data[0]
        log.info('No users on staff with such email: "%s"', email)
