# coding: utf-8


import math
import random
import itertools

import jsonfield
from dir_data_sync.models import Organization
from dir_data_sync.org_ctx import get_org
from django.conf import settings
from django.core.urlresolvers import reverse
from django.db import models
from django.utils.functional import cached_property
from django.contrib.postgres.fields import ArrayField, JSONField

from model_utils import Choices
from model_utils.models import TimeStampedModel

from intranet.dogma.dogma.utils import switch_db_to, get_slave_or_master
from intranet.dogma.dogma.core.errors.utils import get_error_value


class ObjectWithErrorModel(TimeStampedModel):
    last_sync_fail_error_code = models.CharField(
        max_length=50,
        null=True,
        blank=True,
    )

    @cached_property
    def error_value(self):
        if self.last_sync_fail_error_code:
            return get_error_value(self.last_sync_fail_error_code)
        return ''

    class Meta:
        abstract = True


class Node(TimeStampedModel):
    name = models.CharField(
        max_length=100,
        default='not set',
    )

    hostname = models.CharField(
        max_length=100,
    )
    space_total = models.BigIntegerField(default=0)
    space_available = models.BigIntegerField(default=0)
    enabled = models.BooleanField(default=True)  # включена руками

    def __str__(self):
        return '({}) {}'.format(self.name, self.hostname)

    def is_space_running_short(self):
        critical_space = self.space_total * settings.DOGMA_DISK_SIZE_FACTOR
        return self.space_available < critical_space

    def get_score(self, size_estimation):
        size_factor = settings.DOGMA_DISK_SIZE_FACTOR
        have_space = (self.space_total * size_factor) < (self.space_available - size_estimation * 2)
        used = self.space_total - self.space_available
        return int(have_space) * (self.space_available / (used + size_estimation * 2))


class Source(ObjectWithErrorModel):
    VCS_TYPES = Choices(
        'git',
        'github',
        'git_stash',
        'gitlab',
        'svn',
        'hg',
        'bzr',
        'cvs',
        'gerrit',
        'arc',
    )
    WEB_TYPES = Choices(
        'github',
        'stash',
        'websvn',
        'gitweb',
        'hgweb',
        'stash',
        'gitlab',
        'bitbucket_ext',
        'bitbucket_yateam',
        'gerrit',
        'arc',
    )
    WEB_AUTH_TYPES = Choices(
        'basic',
        'cookie',
        'token',
        'x_oauth_token',
        'none',
    )
    VCS_PROTOCOL_TYPES = Choices(
        'ssh',
        'https',
        'http',
        'native',
    )
    SYNC_STATUSES = Choices(
        'success',
        'fail',
    )

    name = models.CharField(
        max_length=100,
    )
    code = models.SlugField()
    vcs_type = models.CharField(
        max_length=20,
        choices=VCS_TYPES,
    )
    web_type = models.CharField(
        max_length=20,
        choices=WEB_TYPES,
    )
    web_url = models.CharField(
        max_length=200,
        help_text='Не используйте слеш "/" в конце строки'
    )
    host = models.CharField(
        max_length=100,
    )
    web_auth = models.CharField(
        max_length=16,
        choices=WEB_AUTH_TYPES,
        default=WEB_AUTH_TYPES.basic
    )
    vcs_protocol = models.CharField(
        max_length=16,
        choices=VCS_PROTOCOL_TYPES,
        default=VCS_PROTOCOL_TYPES.native
    )
    rate = models.DecimalField(default=0, max_digits=3, decimal_places=2)
    ratio = models.PositiveSmallIntegerField(default=1)
    hidden = models.BooleanField(default=False)
    use_crawler = models.BooleanField(
        default=True,
        help_text='Снимите, если хотите вручную заносить в базу репозитории, не используя паука'
    )
    extra_info = jsonfield.JSONField(
        help_text='Можно доопределить {"basic": "username:password", "token": "token"}',
        null=True,
        blank=True,
    )
    total_commits = models.IntegerField(
        default=0, help_text='Комитов в этом источнике'
    )

    status = models.CharField(
        max_length=20,
        default=SYNC_STATUSES.fail
    )

    last_sync_success_time = models.DateTimeField(null=True)
    last_sync_fail_time = models.DateTimeField(null=True)

    last_sync_fail_trace = models.CharField(
        max_length=4000,
        null=True
    )


    def __str__(self):
        return self.code

    def get_absolute_url(self):
        return reverse('dashboard:source_details', args=[str(self.id)])

    def get_rate_delta(self):
        """
        Сколько еще надо склонировать репозиториев из источника.

        @rtype: int
        """
        total = self.repo_set.count()
        if total == 0:
            return 0
        cloned = self.repo_set.filter(clones__isnull=False).distinct().count()
        delta = int(math.ceil(total * (float(self.rate) - (float(cloned) / total))))

        return delta if delta > 0 else 0

    @cached_property
    def crawler(self):
        from .crawlers import get_crawler

        return get_crawler(self)

    @cached_property
    def repos_count(self):
        return self.repo_set.count()

    @cached_property
    def clones_count(self):
        return Clone.objects.filter(repo__source=self, status=Clone.STATUSES.active).count()

    @cached_property
    def failed_clones_count(self):
        return Clone.objects.filter(repo__source=self, status=Clone.STATUSES.fail).count()

    @cached_property
    def clones_pct(self):
        return int(float(self.clones_count) / (self.repos_count or 1) * 100)

    @cached_property
    def failed_clones_pct(self):
        return int(float(self.failed_clones_count) / (self.clones_count or 1) * 100)

    def update_commits_pct(self):
        from .dao.commits import update_commits_count
        update_commits_count(self)

    @cached_property
    def need_create_commits(self):
        return self.web_type != self.WEB_TYPES.gerrit


class ObjectWithOrganizationManager(models.Manager):
    def get_objects_for_current_org(self):
        qs = super(ObjectWithOrganizationManager, self).get_queryset()
        if settings.IS_BUSINESS:
            qs = qs.filter(connect_organization=get_org()).prefetch_related('connect_organization')
        return qs


class Repo(ObjectWithErrorModel):
    SYNC_STATUSES = Choices(
        'success',
        'fail',
        'new',
    )
    source = models.ForeignKey(Source)
    parent = models.ForeignKey('self', null=True, related_name='forks', blank=True)
    vcs_name = models.CharField(
        max_length=250,
    )
    name = models.CharField(
        max_length=250,
    )
    owner = models.CharField(
        max_length=100,
    )
    description = models.TextField(
        default='',
    )
    default_branch = models.CharField(
        max_length=255, default='master',
    )
    is_public = models.BooleanField(
        default=True,
        help_text='Доступен ли репозиторий всем, или доступ к нему ограничен'
    )
    on_remote = models.BooleanField(
        default=True,
        help_text='Есть ли репозиторий на удаленном источнике'
    )
    commits_in_intrasearch = models.BigIntegerField(
        default=0, help_text='Комитов запушено в поиск'
    )
    known_commits = models.BigIntegerField(
        default=0, help_text='Всего известно комитов'
    )

    contiguous_chain_of_commits_ends_at = models.ForeignKey(
        'PushedCommit', null=True,
        help_text='Последний известный коммит',
        related_name='repo_last_known_commit'
    )

    status = models.CharField(
        max_length=20,
        default=SYNC_STATUSES.new,
        db_index=True
    )

    last_sync_success_time = models.DateTimeField(null=True)
    last_sync_fail_time = models.DateTimeField(null=True)

    last_sync_fail_trace = models.CharField(
        max_length=4000,
        null=True
    )

    last_yt_sync_time = models.DateTimeField(null=True)

    do_not_delete = models.BooleanField(default=False, help_text='Крупный репозиторий, который не следует удалять')
    is_important = models.BooleanField(
        default=False,
        help_text='Важный репозиторий, вся работа с ним ведется в отдельной очереди',
    )
    credentials = models.ManyToManyField('Credential', blank=True, )

    is_active = models.BooleanField(default=True, help_text='Есть ли коммиты за последние полгода')
    create_commits_needed = models.BooleanField(default=False)
    create_commits_always = models.BooleanField(default=False)

    url = models.URLField(max_length=1000, null=True, blank=True, )
    organisation = models.ForeignKey('OrganisationsToClone', null=True,
                                     blank=True, related_name='repo',
                                     )
    connect_organization = models.ManyToManyField(Organization, blank=True, )

    objects = ObjectWithOrganizationManager()

    clone_attempt = models.IntegerField(
        default=0, help_text='Неуспешных попыток клонирования',
    )

    sync_delay = models.PositiveSmallIntegerField(default=1)

    class Meta:
        unique_together = [('source', 'owner', 'name')]

    def __str__(self):
        return self.full_name

    def get_credentials(self, auth_type):
        org_credentials = tuple()
        if self.organisation:
            org_credentials = self.organisation.credentials.filter(auth_type=auth_type)
        credentials = sorted(
            itertools.chain(self.credentials.filter(auth_type=auth_type),
                            org_credentials,
                            ),
            key=lambda credential: credential.is_success,
            reverse=True,
        )
        return credentials

    def get_absolute_url(self):
        return reverse('dashboard:repo_details', args=[str(self.id)])

    def get_admin_url(self):
        return reverse('admin:{}_{}_change'.format(self._meta.app_label,
                                                   self._meta.model_name),
                       args=[str(self.id)])

    @property
    def full_name(self):
        return '{source}:{owner}/{name}'.format(
            source=self.source.code, owner=self.owner, name=self.name
        )

    @cached_property
    def backend(self):
        from .backends import get_backend

        return get_backend(self)

    def get_matching_node(self, size_estimation=0):
        """
        size_estimation - примерный размер репозитория в байтах

        возвращает ноду на которой нужно клонировать репозиторий, или None,
        если на нодах нет места.

        """
        used_nodes = list(self.clones.values_list('node_id', flat=True))

        scores = sorted(((node, node.get_score(size_estimation))
                        for node in Node.objects.exclude(id__in=used_nodes, enabled=False)), key=lambda x: x[1], reverse=True)

        scores = scores[:max(self.source.ratio - len(used_nodes), 0)]

        try:
            node, max_score = scores[0]
        except IndexError:
            return
        else:
            if max_score == 0:
                return

        # берем ноды на которых скор не отличается от max_score больше чем на 15%
        diff = [(nd, max_score - scr) for nd, scr in [_f for _f in scores if _f]]
        candidats = [x for x in diff if (x[1] / max_score) <= 0.15]

        node, score = random.choice(candidats)

        return node

    def update_commits_statistics(self):
        from intranet.dogma.dogma.core.dao.commits import update_commits_stats_for_repo
        db_to_use = get_slave_or_master()
        with switch_db_to(db_to_use):
            update_commits_stats_for_repo(self)
        self.save()

    def after_credentials_change(self):
        from intranet.dogma.dogma.core.tasks import clone_repo, fetch_clone
        from intranet.dogma.dogma.core.utils import get_random_node_queue, get_node_queue
        clones = self.clones.all()
        if clones:
            for clone in clones:
                fetch_clone.apply_async(
                    args=[clone.id],
                    queue=get_node_queue('clone', clone.node),
                )
        else:
            clone_repo.apply_async(
                args=[self.id],
                queue=get_random_node_queue('clone'),
            )
        self.clone_attempt = 0
        self.save()


class CloneManager(models.Manager):
    def on_current_node(self):
        from .utils import get_current_node

        return self.get_queryset().filter(node=get_current_node())


class Clone(TimeStampedModel):
    STATUSES = Choices(
        'new',
        'active',
        'fail',
    )

    repo = models.ForeignKey(Repo, related_name='clones')
    node = models.ForeignKey(Node, related_name='clones')
    status = models.CharField(
        max_length=20,
        choices=STATUSES,
    )
    space_required = models.BigIntegerField(default=0)
    path = models.CharField(
        max_length=1000,
    )
    commits_count = models.IntegerField(default=0)
    last_pushed_commit = models.CharField(max_length=250, default='')

    objects = CloneManager()

    class Meta:
        unique_together = [('repo', 'node')]

    def get_absolute_url(self):
        return reverse('dashboard:clone_details', args=[str(self.id)])

    def __str__(self):
        return '{} {}'.format(self.repo, self.node)


class FetchEvent(TimeStampedModel):
    clone = models.ForeignKey(Clone)

    def __str__(self):
        return '{} {}'.format(
            self.clone,
            self.created.isoformat() if self.created else 'new'
        )


class BranchStamp(models.Model):
    fetch_event = models.ForeignKey(FetchEvent)
    name = models.CharField(max_length=128, db_index=True)
    head = models.CharField(max_length=40)
    head_time = models.DateTimeField()
    head_message = models.CharField(max_length=1024)

    class Meta:
        unique_together = [('fetch_event', 'name')]

    def __str__(self):
        return '{} {}'.format(self.fetch_event, self.name)


class User(models.Model):
    uid = models.CharField(max_length=16, db_index=True, null=True)
    login = models.CharField(max_length=50, db_index=True, default='')
    email = models.CharField(max_length=1000, null=True)
    name = models.CharField(max_length=1000)
    other_emails = models.CharField(max_length=1000, help_text='Не рабочие email-адреса через запятую', default='')
    from_staff = models.BooleanField(default=True, help_text='Подтвержденный пользователь со стаффа')

    @cached_property
    def other_emails_parsed(self):
        if not self.other_emails:
            return []
        return set(self.other_emails.split(','))

    @cached_property
    def other_emails_usernames(self):
        return set([
            email.split('@')[0] for email in self.other_emails_parsed
        ])

    @cached_property
    def for_push(self):
        data = {
                'login': self.login,
                'name': self.name,
                'email': self.email,
                'is_trusted_login': self.from_staff,
        }
        if not settings.IS_BUSINESS:
            data['uid'] = self.uid

        return data

    def __str__(self):
        return '{} "{}"-({})'.format(self.id, self.login, self.email)


class OrganisationsToClone(models.Model):
    source = models.ForeignKey(Source)
    name = models.CharField(max_length=100, help_text='Имя организации')
    credentials = models.ManyToManyField('Credential', blank=True, )
    connect_organization = models.ManyToManyField(Organization, blank=True, )
    is_active = models.BooleanField(default=True)

    objects = ObjectWithOrganizationManager()

    def __str__(self):
        return '{}'.format(self.name)

    def get_credentials(self, auth_type):
        credentials = sorted(
            self.credentials.filter(auth_type=auth_type),
            key=lambda credential: credential.is_success,
            reverse=True,
        )
        return credentials

    def make_active(self):
        if not self.is_active:
            self.is_active = True
            self.save()
        self.repo.filter(is_active=False).update(is_active=True)



class BaseCommit(TimeStampedModel):
    TYPES = Choices(
        'common',
        'merge',
    )
    repo = models.ForeignKey(Repo)
    author = models.ForeignKey(User, related_name='%(class)s_author')
    committer = models.ForeignKey(User, related_name='%(class)s_committer')
    parents = models.ManyToManyField("self")
    commit_time = models.DateTimeField()
    lines_added = models.PositiveIntegerField()
    lines_deleted = models.PositiveIntegerField()
    message = models.TextField()
    branch_name = models.CharField(max_length=500)
    tree_hex = models.CharField(max_length=40)
    commit_id = models.CharField(max_length=40)
    commit_type = models.CharField(
        max_length=20,
        choices=TYPES,
        default=TYPES.common,
    )

    tickets = ArrayField(models.CharField(max_length=100, blank=True), blank=True, null=True, db_index=True)
    queues = ArrayField(models.CharField(max_length=50, blank=True), blank=True, null=True, db_index=True)

    def __str__(self):
        return '{hex} {commit_time}'.format(
            hex=self.commit,
            commit_time=self.commit_time,
        )

    class Meta:
        abstract = True


class PushedCommit(BaseCommit):
    commit = models.CharField(max_length=50, unique=True)
    aggregated = models.NullBooleanField(null=True, blank=True)
    # TODO Убрать код ниже после создания данных о файлах всех существующих коммитов
    create_changed_files = models.NullBooleanField(null=True, blank=True)


class PushedCommitDuplicate(BaseCommit):
    commit = models.CharField(max_length=50)


class PushedCommitDuplicateHex(models.Model):
    commit = models.CharField(max_length=50)
    repo = models.ForeignKey(Repo)


class ChangedFile(models.Model):
    commit = models.ForeignKey('PushedCommit', related_name='changed_files')
    name = models.CharField(max_length=500)
    extension = models.CharField(max_length=100)
    lines_added = models.PositiveIntegerField()
    lines_deleted = models.PositiveIntegerField()


class UserFileStatistics(models.Model):
    user = models.ForeignKey(User)
    extension = models.CharField(max_length=100)
    lines_added = models.PositiveIntegerField()
    lines_deleted = models.PositiveIntegerField()
    period = models.DateField()


class Credential(TimeStampedModel):
    AUTH_TYPES = Choices(
        'token',
        'app_password',
    )
    name = models.CharField(max_length=250)
    auth_type = models.CharField(
        max_length=100,
        choices=AUTH_TYPES,
    )
    auth_data = JSONField()
    connect_organization = models.ForeignKey(Organization)
    is_success = models.BooleanField(default=True)

    objects = ObjectWithOrganizationManager()

    def mark_fail(self):
        if self.is_success:
            self.is_success = False
            self.save()

    def mark_success(self):
        if not self.is_success:
            self.is_success = True
            self.save()

    def get_related_repos(self):
        related_repos = set(self.repo_set.all().prefetch_related('clones'))
        for org in self.organisationstoclone_set.all():
            related_repos.update(org.repo.all().prefetch_related('clones'))

        return related_repos

    def __str__(self):
        return '{name} - {connect_organization}'.format(
            name=self.name,
            connect_organization=self.connect_organization,
        )
