# coding: utf-8


import socket
import functools
import logging
from random import random
from bisect import bisect_right
import numpy as np
import ylock

from django.conf import settings
from django.core.cache import cache

from intranet.dogma.dogma.core.models import Node
from .git.models import Repository as GitRepository
from .hg.models import Repository as HgRepository
from .svn.models import Repository as SvnRepository
from .arcc.models import Repository as ArcRepository

logger = logging.getLogger(__name__)

NODE_PATTERN = '%s@%s.dq2'


def get_current_node(create_missing=False, no_cache=False):
    hostname = socket.getfqdn()
    key = 'curnode_%s' % hostname

    if no_cache:
        node = None
    else:
        node = cache.get(key, None)

    if node is not None:
        return node

    try:
        node = Node.objects.get(hostname=hostname)
    except Node.DoesNotExist:
        if create_missing:
            node = Node.objects.create(
                hostname=hostname, space_total=0, space_available=0,
            )
        else:
            raise

    node._state.db = None       # для репликейтеда
    cache.set(key, node)
    return node


def get_node_queue(name='celery', node=None):
    return NODE_PATTERN % (name, (node or get_current_node()).hostname)


def get_random_node_queue(name='celery'):
    node = Node.objects.filter(enabled=True).order_by('?').values_list('hostname',
                                                                       flat=True)[0]
    return NODE_PATTERN % (name, node)


lock_manager = ylock.backends.create_manager(**settings.MANAGER_SETTINGS)


locked_context = functools.partial(
    lock_manager.lock,
    block=False,
)


def locked(lock_name):
    def _decorator(func):
        @functools.wraps(func)
        def _wrapper(*args, **kwargs):
            with locked_context(lock_name) as is_locked:
                if is_locked:
                    return func(*args, **kwargs)
                else:
                    logger.info('Lock "%s" is acquired by another process',
                                lock_name,
                                )
        return _wrapper
    return _decorator


def get_repository_model(clone):
    """
    Вернуть нужную модель репозитория в зависимости от клона.

    Либо hg, либо git.

    @type clone: Clone
    """
    if clone.repo.source.vcs_type == 'hg':
        return HgRepository.discover(clone.path)
    elif clone.repo.source.vcs_type == 'svn':
        return SvnRepository.discover(clone.path)
    elif clone.repo.source.vcs_type == 'arc':
        return ArcRepository.discover(clone)
    return GitRepository.discover(clone.path)


def weighted_shuffle(sequence, weights):
    result = np.empty_like(sequence)
    cum_weights = np.cumsum(weights)
    for i in range(len(sequence)):
        rnd = random() * cum_weights[-1]
        j = bisect_right(cum_weights, rnd)
        result[i] = sequence[j]
        cum_weights[j:] -= weights[j]
    return result
