from collections import OrderedDict
from functools import wraps
import datetime

from django_yauth.decorators import yalogin_required

from django.core.paginator import Paginator, PageNotAnInteger, EmptyPage
from django.core.urlresolvers import reverse
from django.http import HttpResponseRedirect, HttpResponse, Http404
from django.shortcuts import render, get_object_or_404, redirect
from django.views.decorators.http import require_POST
from django.contrib import messages
from django import forms
from django.db.models import Count, Case, When
from django.db.models.query_utils import Q
from django_tools_log_context import request_context, request_profiler

from intranet.dogma.dogma.core.utils import get_node_queue, get_random_node_queue
from intranet.dogma.dogma.core.tasks import (
    clone_repo, fetch_clone,
    clone_source, update_source,
    repair_failed, update_clones,
)
from intranet.dogma.dogma.core.models import Node, Source, Repo, Clone
from intranet.dogma.dogma.core.dao.repo import get_straggled_repos, get_failed_repos, select_nodes_for_repo
from intranet.dogma.dogma.core.dao.commits import get_commits_stat_by_users_by_months


def add_ctx(view_func):
    @wraps(view_func)
    def wrapped(request, *args, **kwargs):
        with request_context(request, endpoint=view_func), request_profiler(request, threshold=5):
            return view_func(request, *args, **kwargs)
    return wrapped


@add_ctx
@yalogin_required
def index(request):
    return HttpResponseRedirect(reverse('dashboard:sources_list'))


@add_ctx
@yalogin_required
def sources_list(request):
    objects = Source.objects.all()

    commits = sum((source.total_commits for source in objects))

    return render(request, 'dashboard/sources_list.html', {
        'objects': objects,
        'commits': commits,
        'actions': ['parse_source', 'repair_failed_clones', ],
    })


@add_ctx
@require_POST
@yalogin_required
def source_details(request, pk, action=None):
    obj = get_object_or_404(Source, pk=pk)

    if action == 'parse_source':
        update_source.apply_async(args=(obj.id, ),
                                  queue=get_random_node_queue('clone'),
                                  )
    elif action == 'repair_failed_clones':
        repair_failed.apply_async(args=[[obj.id]],
                                  queue=get_random_node_queue('clone'),
                                  )
    else:
        messages.error(request, "Unknown action")
        return HttpResponse('{"result": "Bad request"}', status=400)

    messages.success(request, "Repositories was queued for %s" % action)
    return HttpResponse('{"result": "Done"}')


class UpdateSourceForm(forms.Form):
    action_type = forms.ChoiceField(
        choices=(('Fetch', 'Fetch'),
                 ('Clone', 'Clone'),
                 ),
    )


@add_ctx
@require_POST
@yalogin_required
def source_update_by_type(request, pk):
    f = UpdateSourceForm(request.POST)

    if f.is_valid():
        action_type = f.cleaned_data['action_type'].lower()

        if action_type == 'fetch':
            action = update_clones
        elif action_type == 'clone':
            action = clone_source

        action.apply_async(kwargs={'source_id': pk},
                           queue=get_random_node_queue('clone'),
                           )
        messages.success(request, "Repositories was queued for %s" %
                         action_type)
    else:
        messages.warning(request, "Bad request")

    return redirect('dashboard:sources_list')


@add_ctx
@yalogin_required
def repos_list(request):
    search_string = request.GET.get('search_string', '')
    objects = Repo.objects.order_by('owner', 'name')

    if search_string:
        objects = objects.filter(
            Q(owner__icontains=search_string) |
            Q(name__icontains=search_string) |
            Q(source__code__icontains=search_string),
        )

    paginator = Paginator(objects, request.GET.get('limit', 51))
    try:
        page = paginator.page(request.GET.get('page', 1))
    except PageNotAnInteger:
        page = paginator.page(1)
    except EmptyPage:
        page = paginator.page(paginator.num_pages)

    if request.method == 'POST' and search_string:
        for obj in objects:
            acceptable_volumes = select_nodes_for_repo(obj)
            if len(acceptable_volumes) > 0:
                volume = acceptable_volumes[0]
                clone_repo.apply_async(args=(obj.id,), queue=get_node_queue('clone', volume))
            else:
                messages.error(request, "No acceptable volumes found")

        else:
            messages.success(request, "Filtred repositories queued on cloning")

    return render(request, 'dashboard/repos_list.html', {
        'objects': page,
        'search_string': search_string,
    })


@add_ctx
@yalogin_required
def failed_repos(request):
    action = request.POST.get('action')
    important_only = request.GET.get('important')

    if request.method == 'POST' and action:
        if action == 'repair_repos':
            repair_failed.apply_async(queue=get_random_node_queue('clone'))
            messages.success(request, "Starting to repair failed repos")

    objects = get_failed_repos(important=important_only)
    ordered_objects = objects.order_by('-last_sync_fail_time').select_related()

    paginator = Paginator(ordered_objects, request.GET.get('limit', 51))
    try:
        page = paginator.page(request.GET.get('page', 1))
    except PageNotAnInteger:
        page = paginator.page(1)
    except EmptyPage:
        page = paginator.page(paginator.num_pages)
    return render(request, 'dashboard/failed_repos.html', {
        'objects': page,
        'failed_count': paginator.count
    })


@add_ctx
@yalogin_required
def straggled_repos(request):
    important_only = request.GET.get('important')

    objects = get_straggled_repos(important=important_only)
    objects = objects.order_by('last_sync_success_time').select_related()[:1000]

    return render(request, 'dashboard/straggled_repos.html', {
        'objects': objects,
    })


@add_ctx
@yalogin_required
def repo_details(request, pk, action=None):
    try:
        obj = Repo.objects.prefetch_related('clones').get(pk=pk)
    except Repo.DoesNotExist:
        raise Http404()

    if request.method == 'POST' and request.is_ajax() and action:
        if action == 'clone':
            acceptable_volumes = select_nodes_for_repo(obj)
            if len(acceptable_volumes) == 0:
                messages.error(request, 'No acceptable volumes found')
            else:
                volume = acceptable_volumes[0]
                clone_repo.apply_async(args=(obj.id,), queue=get_node_queue('clone', volume))
                messages.success(request, "Repository was queued for %s" % action)

        elif action == 'update':
            for clone in obj.clones.select_related('node'):
                fetch_clone.apply_async(args=(clone.id,),
                                        queue=get_node_queue('clone', clone.node))

            messages.success(request, "Clone was queued for %s" % action)
        elif action == 'set on remote':
            obj.on_remote = True
            obj.save()
            messages.success(request, "Successfully set On remote - True")
        elif action == 'set is active':
            obj.is_active = True
            obj.save()
            messages.success(request, "Successfully set Is active - True")
        else:
            messages.error(request, "Unknown action %s" % action)

        return HttpResponse('{"result": "KOKOKO"}')

    return render(request, 'dashboard/repo_details.html', {
        'objects': Repo.objects.order_by('owner', 'name'),
        'object': obj,
        'actions': ['update', 'clone', 'set on remote', 'set is active', ],
        'clone_actions': ['update', ],
    })


@add_ctx
@yalogin_required
def nodes_list(request):
    objects = Node.objects.annotate(
        failed_clones=Count(Case(When(clones__status=Clone.STATUSES.fail, then=1))),
        active_repos=Count(Case(When(clones__repo__is_active=True, then=1)))
    ).order_by('-enabled', 'name')
    return render(request, 'dashboard/nodes_list.html', {
        'objects': objects,
    })


@add_ctx
@yalogin_required
def clone_details(request, pk, action):
    clone = get_object_or_404(Clone, pk=pk)

    if request.method == 'POST' and request.is_ajax() and action:
        if action == 'update':
            fetch_clone.apply_async(args=(clone.id,),
                                    queue=get_node_queue('clone', clone.node))
            messages.success(request, "Clone %s was queued for update" % clone.id)

        else:
            messages.error(request, "Unknown action")

    return HttpResponse('{"result": "Done"}')


@add_ctx
@yalogin_required
def commits_stat_by_users_by_months(request):

    logins = request.GET.get("logins") or request.GET.get("login") or request.yauser.login

    logins = logins.split(",")[:20]

    res = get_commits_stat_by_users_by_months(logins)

    objects = {
        "logins": logins,
        "months": OrderedDict()
    }

    objects["months"]["Summary"] = dict()

    for login, commits_count, lines_added, lines_deleted, month, year in res:
        month_name = datetime.date(int(year), int(month), 1).strftime('%B %Y')
        if month_name not in objects["months"]:
            objects["months"][month_name] = dict()
        objects["months"][month_name][login] = {
            "commits_count": commits_count,
            "lines_added": int(lines_added),
            "lines_deleted": int(lines_deleted),
        }

        if login not in objects["months"]["Summary"]:
            objects["months"]["Summary"][login] = {
                "commits_count": commits_count,
                "lines_added": int(lines_added),
                "lines_deleted": int(lines_deleted),
            }
        else:
            objects["months"]["Summary"][login]["commits_count"] += commits_count
            objects["months"]["Summary"][login]["lines_added"] += int(lines_added)
            objects["months"]["Summary"][login]["lines_deleted"] += int(lines_deleted)

    for month in objects["months"].keys():
        for login in logins:
            if login not in objects["months"][month]:
                objects["months"][month][login] = {
                    "commits_count": 0,
                    "lines_added": 0,
                    "lines_deleted": 0,
                }

        objects["months"][month] = [objects["months"][month][login] for login in logins]

    objects["months"]["Summary"][0]["show_legend"] = True

    return render(request, 'dashboard/commits_stat_by_users_by_months.html', {
        'objects': objects,
    })

