from base64 import b64decode, b64encode
from urllib.parse import urlencode, urlparse, parse_qsl, urlunparse

from fastapi import Request, status
from sqlalchemy import and_, desc
from starlette.exceptions import HTTPException
from typing import Optional, Callable, Any,     Tuple
from typing import Set

from .schemas.common import CursorPaginationResponse


class CursorPaginator:
    # Включает в себя логику построения ссылок на следующий/предыдущий набор данных
    # для формирования ответа текущий набор данных подставляется извне в метод build_response
    def __init__(
        self,
        cursor: Optional[str],
        limit: int,
        reverse: bool,
        cursor_fields: Set[str],
        cursor_param: str,
        limit_param: str,
        reverse_param: str,
        request: Request,
    ):
        self.cursor = cursor
        self.limit = limit
        self.reverse = reverse
        self.cursor_param = cursor_param
        self.cursor_fields = cursor_fields
        self.limit_parameter = limit_param
        self.reverse_parameter = reverse_param
        self.request = request
        self.cursor_data = None

    def is_cursor_present(self) -> bool:
        return bool(self.cursor)

    def _default_item_cursor_data_getter(self, item) -> dict:
        cursor = {}
        for field in self.cursor_fields:
            cursor[field] = getattr(item, field)
        return cursor

    @staticmethod
    def _encode_cursor(cursor_data: dict) -> str:
        querystring = urlencode(cursor_data, doseq=True)
        return b64encode(querystring.encode('ascii')).decode('ascii')

    @staticmethod
    def _decode_cursor(cursor: str) -> dict:
        querystring = b64decode(cursor.encode('ascii')).decode('ascii')
        decoded = parse_qsl(querystring, keep_blank_values=True)
        return {
            key: value
            for key, value in decoded
        }

    def get_cursor_data(self) -> Optional[dict]:
        if self.cursor_data is None and self.is_cursor_present():
            self.cursor_data = self._decode_cursor(self.cursor)

        return self.cursor_data

    def build_response(
        self,
        items: list,
        item_cursor_data_getter: Optional[Callable[[Any], dict]] = None,
    ) -> CursorPaginationResponse:
        if item_cursor_data_getter is None:
            item_cursor_data_getter = self._default_item_cursor_data_getter

        is_last = len(items) <= self.limit
        if not is_last:
            del items[-1]

        if self.reverse:
            # если запрашиваем записи до курсора, то сортировка происходит по мере удаления от записи,
            # на которую указывает курсор, это позволяет выбрать правильный набор записей с учетом лимита,
            # но в результате список этих записей получится отсортированным в обратную сторону,
            # поэтому список нужно перевернуть
            items.reverse()

        # request_query_params = dict(self.request.query_params)

        next_link = None
        if self.reverse or (not self.reverse and not is_last):
            next_cursor_data = item_cursor_data_getter(items[-1])
            next_cursor = self._encode_cursor(next_cursor_data)
            next_link = str(self.request.url.include_query_params(
                **{self.cursor_param: next_cursor},
            ))

        prev_link = None
        if self.is_cursor_present() and (not self.reverse or (self.reverse and not is_last)):
            prev_cursor_data = item_cursor_data_getter(items[0])
            prev_cursor = self._encode_cursor(prev_cursor_data)
            prev_link = str(self.request.url.include_query_params(
                **{self.cursor_param: prev_cursor, self.reverse_parameter: 1}
            ))

        return CursorPaginationResponse(
            next=next_link,
            prev=prev_link,
            result=items,
        )


class CursorPaginatorFactory:
    def __init__(
        self,
        cursor_fields: Set[str],
        cursor_param: str = 'cursor',
        limit_param: str = 'limit',
        reverse_param: str = 'reverse',
        default_limit: int = 20,
        max_limit: int = 1000,
    ):
        self.cursor_fields = cursor_fields
        self.cursor_param = cursor_param
        self.limit_param = limit_param
        self.reverse_param = reverse_param
        self.default_limit = default_limit
        self.max_limit = max_limit

    def __call__(self, request: Request) -> CursorPaginator:
        limit = get_int_value(request, self.limit_param, self.default_limit)
        if limit > self.max_limit:
            limit = self.max_limit

        reverse = bool(get_int_value(request, self.reverse_param, 0))
        cursor = request.query_params.get(self.cursor_param)

        return CursorPaginator(
            cursor=cursor,
            cursor_param=self.cursor_param,
            limit=limit,
            reverse=reverse,
            cursor_fields=self.cursor_fields,
            limit_param=self.limit_param,
            reverse_param=self.reverse_param,
            request=request,
        )


class CursorModelPaginator:
    # Включает в себя логику получения набора моделей, используя параметры из экземпляра CursorPaginator
    # после получения данных, передает их в экземпляр CursorPaginator для построения ответа
    def __init__(
        self,
        model,
        paginator: CursorPaginator,
        order: str,
        cursor_field: Tuple[str, Callable[[str], Any]],
    ):
        self.model = model
        self.paginator = paginator
        self.order = order
        self.cursor_field = cursor_field

    async def build_response(self, where_filter=None):
        model_cursor_field = getattr(self.model, self.cursor_field[0])

        wheres = []
        if self.paginator.is_cursor_present():
            field, caster = self.cursor_field
            value = caster(self.paginator.get_cursor_data()[field])
            if (not self.paginator.reverse and self.order == 'asc') or (self.paginator.reverse and self.order == 'desc'):
                wheres.append(model_cursor_field > value)
            else:
                wheres.append(model_cursor_field < value)

        if (not self.paginator.reverse and self.order == 'asc') or (self.paginator.reverse and self.order == 'desc'):
            order_by = model_cursor_field
        else:
            order_by = desc(model_cursor_field)

        query = getattr(self.model, 'query')

        items = await query \
            .where(and_(*wheres, where_filter)) \
            .order_by(order_by) \
            .limit(self.paginator.limit + 1) \
            .gino.all()

        return self.paginator.build_response(items)


class CursorModelPaginatorFactory:
    def __init__(
        self,
        model,
        cursor_field: Optional[Tuple[str, Callable[[str], Any]]] = None,
        cursor_param: str = 'cursor',
        limit_param: str = 'limit',
        reverse_param: str = 'reverse',
        default_limit: int = 20,
        max_limit: int = 1000,
        order: str = 'asc',
    ):
        if cursor_field is None:
            cursor_field = ('id', int)
        self.cursor_field = cursor_field

        self.factory = CursorPaginatorFactory(
            cursor_fields={cursor_field[0]},
            cursor_param=cursor_param,
            limit_param=limit_param,
            reverse_param=reverse_param,
            default_limit=default_limit,
            max_limit=max_limit,
        )

        self.model = model

        self.order = order.lower()
        if self.order not in {'asc', 'desc'}:
            raise ValueError('order must be asc or desc')

    def __call__(self, request: Request) -> CursorModelPaginator:
        paginator = self.factory(request)
        return CursorModelPaginator(self.model, paginator, self.order, self.cursor_field)


def add_params_to_url(url, query_params):
    url_parts = list(urlparse(url))
    query = dict(parse_qsl(url_parts[4]))
    query.update(query_params)

    url_parts[4] = urlencode(query)
    return urlunparse(url_parts)


def get_int_value(request: Request, param_name: str, default: int):
    str_value = request.query_params.get(param_name, default)
    try:
        return int(str_value)
    except ValueError:
        raise HTTPException(
            status.HTTP_422_UNPROCESSABLE_ENTITY,
            'Parameter \'{param_name}\' must be of type int'.format(
                param_name=param_name)
        )
