from datetime import datetime, timedelta
from async_clients.exceptions.base import BadResponseStatus
from async_clients.exceptions.gendarme import (
    DomainNotFoundException
)
from async_clients.exceptions.webmaster import (
    WebmasterException as WebmasterClientException,
    UnAllowedVerificationType as UnAllowedClientVerificationType,
)

from intranet.domenator.src.logic.exceptions.regru import RegruException

from fastapi import APIRouter, BackgroundTasks, status, Depends
from sqlalchemy import and_, desc
from starlette.exceptions import HTTPException
from typing import Optional, List, Set

from intranet.domenator.src.logic.connect import get_connect_client
from intranet.domenator.src.logic.utils import (
    chunks,
    parse_to_datetime,
)
from intranet.domenator.src.db import Domain, MailUserDomainStatus, MailUserDomain
from intranet.domenator.src.logic.exceptions.webmaster import (
    WebmasterException,
    UnAllowedVerificationType,
)
from intranet.domenator.src.logic.mailsettings import get_mailsettings_client
from intranet.domenator.src.logic.gendarme import get_gendarme_client
from intranet.domenator.src.logic.webmaster import (
    get_domain_ownership_info,
    get_webmaster_client,
)
from ..common import ListParser
from ..pagination import (
    CursorPaginationResponse,
    CursorModelPaginator,
    CursorModelPaginatorFactory,
)
from ..schemas.domain import (
    WhoIsOutSchema,
    OwnershipInfoOutSchema,
    CheckOwnershipOutSchema,
    CheckOwnershipInSchema,
    DomainOutSchema,
    DomainHistoryOutSchema,
    DomainAddInSchema,
    DomainSyncInSchema,
    UpdateAdminInSchema,
    OwnedDomainAddInSchema,
)
from ...db.domain.models import DomainAction, DomainHistory
from ...logic.blackbox import get_blackbox_client
from ...logic.domain import (
    is_tech,
    save_domain_history,
    sync_domain_state,
    check_domain_is_correct,
    to_punycode,
    create_domain_in_passport,
    after_first_domain_became_confirmed,
)
from intranet.domenator.src.logic.mail_user_domain import (
    filter_domain_candidates,
    get_valid_domain,
    get_valid_login,
    check_user_can_register_or_throw,
    ACTIVE_MAILUSER_DOMAIN_STATUSES_STR,
    check_login_allowed_or_throw,
    get_candidates,
)
from ...logic.sender import send_cancelled_domain_mail
from ...logic.event import save_event
from ...logic.event.events import DomainDeletedEvent, DomainAddedEvent
from ...logic.passport import delete_domain_in_passport, get_passport_client
from ...logic.regru import get_regru_client, REGRU_ERRORS_BAD_INPUT, REGRU_ERRORS_UNAVAILABLE
from ...settings import config
import logging
from intranet.domenator.src.domenator_logging.logger import get_logger

router = APIRouter()
log = logging.getLogger(__name__)
error_log = get_logger(log_name='dom_error')
default_log = get_logger(log_name='dom_default')


# this function is used for mocking in
# unit tests
def get_now():
    return datetime.now()


@router.get(
    "/who-is/",
    response_model=WhoIsOutSchema,
    response_model_by_alias=False,
)
async def who_is_by_domain(domain: str):
    return await Domain.query.where(
        and_(
            Domain.name == domain,
            Domain.owned == True,  # noqa
        )
    ).gino.first_or_404()


@router.get(
    "/{org_id}/{domain_name}/ownership-info/",
    response_model=OwnershipInfoOutSchema,
)
async def domain_ownership_info(org_id: str, domain_name: str, background_tasks: BackgroundTasks):
    lower_domain_name = domain_name.lower()
    domain = await Domain.query.where(
        and_(
            Domain.name == lower_domain_name,
            Domain.org_id == org_id,
        )
    ).gino.first_or_404()

    result = {
        'domain': domain.name,
        'status': 'need-validation',
        'preferred_host': domain.name,
    }

    if domain.owned:
        result['status'] = 'owned'
    else:
        try:
            status, methods, last_check, preferred_host, verification_status = await get_domain_ownership_info(
                name=domain.name,
                admin_id=domain.admin_id,
            )
        except (WebmasterClientException, BadResponseStatus) as e:
            raise WebmasterException(
                webmaster_message=repr(e),
            )
        if status:
            result['status'] = status
        result['methods'] = methods
        if last_check:
            result['last_check'] = last_check
        if preferred_host:
            result['preferred_host'] = preferred_host

        # если домен подтвержден в webmaster отметим его подтвержденным и у нас в базе
        if verification_status == 'VERIFIED':
            background_tasks.add_task(
                sync_domain_state,
                domain_name=domain.name,
                org_id=org_id,
                admin_uid=domain.admin_id,
            )

    return result


@router.post(
    "/{org_id}/{domain_name}/check-ownership/",
    response_model=CheckOwnershipOutSchema,
    response_model_by_alias=False,
)
async def domain_check_ownership(
    org_id: str,
    domain_name: str,
    item: CheckOwnershipInSchema,
    background_tasks: BackgroundTasks,
):
    lower_domain_name = domain_name.lower()

    domain: Domain = await Domain.query.where(
        and_(
            Domain.name == lower_domain_name,
            Domain.org_id == org_id,
        )
    ).gino.first_or_404()

    webmaster = await get_webmaster_client()

    # Разобъем способ подтверждения на webmaster и код самого способа подтверждения
    if '.' not in item.verification_type:
        raise HTTPException(
            status.HTTP_422_UNPROCESSABLE_ENTITY,
            'Not valid verification_type',
        )
    via, verification_type = item.verification_type.split('.', 1)
    try:
        await webmaster.verify(
            domain.name,
            domain.admin_id,
            verification_type,
        )
    except UnAllowedClientVerificationType:
        raise UnAllowedVerificationType(verification_type=verification_type)
    except (WebmasterClientException, BadResponseStatus) as e:
        raise WebmasterException(
            webmaster_message=repr(e),
        )
    background_tasks.add_task(
        sync_domain_state,
        org_id=org_id,
        domain_name=domain.name,
        admin_uid=domain.admin_id,
    )

    return domain


@router.get("/")
async def get_domains(
    org_ids: List[str] = Depends(ListParser('org_ids', required=True)),
    name: Optional[str] = None,
    fields: Set[str] = Depends(ListParser('fields', default={'name'}, container=set)),
):
    wheres = [Domain.org_id.in_(org_ids)]
    if name:
        wheres.append(Domain.name == name)

    domains = await Domain.query.where(and_(*wheres)).gino.all()

    result = []
    gendarme = await get_gendarme_client()
    for domain in domains:
        domain.name = domain.name.lower()
        result_model = DomainOutSchema.from_orm(domain)

        if domain.owned:
            if 'delegated' in fields or 'mx' in fields:
                try:
                    data_from_gendarme = await gendarme.status(domain.name)
                    result_model.mx = data_from_gendarme['mx'].get(
                        'match', domain.mx)
                    result_model.delegated = data_from_gendarme['ns'].get(
                        'match', domain.delegated)
                except DomainNotFoundException:
                    result_model.mx = False
                    result_model.delegated = False

        result_model.tech = is_tech(domain.name)

        result.append(result_model.dict(include=fields))

    return result


@router.delete("/{domain_name}")
async def delete_domain(
    domain_name: str,
    org_id: str,
    admin_uid: str,
    author_id: Optional[str] = None,
    delete_blocked: bool = False,
):
    domain_name = domain_name.lower()

    domain = await Domain.query.where(
        Domain.name == to_punycode(domain_name),
    ).gino.first_or_404()

    if domain.master:
        raise HTTPException(
            status.HTTP_422_UNPROCESSABLE_ENTITY,
            f'Domain {domain_name} can\'t be deleted because it is main domain of organization.',
        )

    if bool(domain.blocked_at) and not delete_blocked:
        raise HTTPException(
            status.HTTP_422_UNPROCESSABLE_ENTITY,
            f'Domain {domain_name} is blocked',
        )

    master_domain_id = None
    master_domain: Domain = await Domain.query.where(
        and_(
            Domain.org_id == org_id,
            Domain.master == True,  # noqa
        )
    ).gino.first()

    if master_domain:
        bb_client = await get_blackbox_client()
        master_domain_bb_info = await bb_client.get_domain_info(master_domain.name)

        if master_domain_bb_info:
            master_domain_id = master_domain_bb_info['domain_id']

    await delete_domain_in_passport(
        domain_name=domain_name,
        admin_uid=admin_uid,
        master_domain_id=master_domain_id,
    )
    await domain.delete()
    await save_domain_history(domain, DomainAction.domain_deleted, author_id)
    await save_event(DomainDeletedEvent(
        domain=domain.name,
        org_id=domain.org_id,
        author_id=admin_uid,
    ))


@router.get(
    "/{domain_name}/history",
    response_model=CursorPaginationResponse[DomainHistoryOutSchema],
)
async def get_domain_history(
    domain_name: str,
    paginator: CursorModelPaginator = Depends(
        CursorModelPaginatorFactory(
            model=DomainHistory,
            order='desc',
        )
    ),
):
    where_filter = DomainHistory.name == domain_name
    return await paginator.build_response(where_filter=where_filter)


@router.post("/")
async def add_domain(data: DomainAddInSchema, background_tasks: BackgroundTasks):
    await check_domain_is_correct(data.org_id, data.domain, data.admin_uid)
    domain: Domain = await Domain.create(
        org_id=data.org_id,
        name=data.domain,
        owned=False,
        admin_id=data.admin_uid,
        via_webmaster=True,
    )
    await save_domain_history(domain, DomainAction.domain_added, data.admin_uid)
    await save_event(DomainAddedEvent(
        domain=domain.name,
        org_id=domain.org_id,
        author_id=data.admin_uid,
    ))

    background_tasks.add_task(
        sync_domain_state,
        domain_name=domain.name,
        org_id=data.org_id,
        admin_uid=data.admin_uid,
        registrar_id=None
    )


@router.post("/add-owned")
async def add_owned_domain(data: OwnedDomainAddInSchema):
    existent_domain = await Domain.query.where(
        and_(
            Domain.name == data.domain,
            Domain.owned == True,  # noqa
        )
    ).gino.first()
    if existent_domain:
        raise HTTPException(
            status.HTTP_422_UNPROCESSABLE_ENTITY,
            'domain_occupied',
        )

    await check_domain_is_correct(data.org_id, data.domain, data.admin_id)

    domain: Domain = await Domain.create(
        org_id=data.org_id,
        name=data.domain,
        owned=False,
        admin_id=data.admin_id,
        via_webmaster=True,
    )
    await save_domain_history(domain, DomainAction.domain_added, data.admin_id)
    await save_event(DomainAddedEvent(
        domain=domain.name,
        org_id=domain.org_id,
        author_id=data.admin_id,
    ))

    is_master = await create_domain_in_passport(data.org_id, data.domain, data.admin_id)
    await domain.update(
        owned=True,
        validated_at=datetime.utcnow(),
    ).apply()

    if is_master:
        await after_first_domain_became_confirmed(data.org_id, domain, data.admin_id)


@router.post("/{org_id}/update-admin")
async def update_admin(org_id: str, data: UpdateAdminInSchema):
    domains: List[Domain] = await Domain.query.where(and_(
        Domain.org_id == org_id,
        Domain.owned == True,  # noqa
    )).order_by(desc(Domain.master)).gino.all()

    blackbox = await get_blackbox_client()
    passport = await get_passport_client()

    for domain in domains:
        domain_bb_info = await blackbox.get_domain_info(domain.name)
        await passport.domain_edit(domain_bb_info['domain_id'], {'admin_uid': data.admin_id})

    await Domain.update.values(admin_id=data.admin_id).where(Domain.org_id == org_id).gino.status()


@router.post("/sync-connect")
async def sync_domains_with_connect(data: DomainSyncInSchema):
    org_id = data.org_id
    admin_id = data.admin_id
    connect = await get_connect_client()
    domains = await connect.get_domains(org_id)
    for chunk in chunks(domains, 50):
        data = []
        for domain in chunk:
            already_exists = await Domain.query.where(
                and_(
                    Domain.name == domain['name'],
                    Domain.org_id == org_id,
                )
            ).gino.first()
            if not already_exists:
                data.append(
                    {
                        'org_id': org_id,
                        'admin_id': admin_id,
                        'name': domain['name'],
                        'master': domain['master'],
                        'owned': domain['owned'],
                        'validated': domain['validated'],
                        'mx': domain['mx'],
                        'delegated': domain['delegated'],
                        'via_webmaster': domain['via_webmaster'],
                        'created_at': parse_to_datetime(domain['created_at']),
                        'validated_at': parse_to_datetime(domain['validated_at']),
                    }
                )
        if data:
            await Domain.insert().gino.all(data, read_only=False, reuse=False)

    return {'status': 'ok'}


@router.get("/suggest/{uid}")
async def suggest_domains(uid: str, ip: str, domain_base: Optional[str] = None, login: Optional[str] = None,
                          limit: Optional[int] = None):
    blackbox = await get_blackbox_client()
    user = await blackbox.get_userinfo(uid, ip)

    if user is None:
        raise HTTPException(
            status.HTTP_404_NOT_FOUND,
            'user not found',
        )

    if login:
        login = get_valid_login(login)
        if not login:
            login = 'me'

    candidates = get_candidates(user, login, domain_base)
    candidates = await filter_domain_candidates(user, candidates, limit)

    suggested_domains = [{
        'login': candidate.login,
        'name': candidate.validated_domain,
    } for candidate in candidates]

    result = {
        'suggested_domains': suggested_domains
    }

    # if domain_base is provided
    if domain_base:
        domain_requested = get_valid_domain(domain_base)
        if domain_requested not in [candidate.validated_domain for candidate in candidates]:
            result['domain_status'] = 'occupied'
        elif domain_requested is None:
            result['domain_status'] = 'not_allowed'
        else:
            result['domain_status'] = 'available'

    return result


@router.post("/register/{uid}")
async def register_domain(uid: str, ip: str, domain: str, login: str):
    await check_user_can_register_or_throw(uid, domain)

    login = get_valid_login(login.strip().lower())
    check_login_allowed_or_throw(login)
    domain = domain.lower()

    regru_client = await get_regru_client()
    domain_record: MailUserDomain = await MailUserDomain.create(
        domain=domain,
        uid=uid,
        status=MailUserDomainStatus.pending_registrar.value,
        login=login
    )

    if uid in config.users_with_emulation:
        await domain_record.update(
            status=MailUserDomainStatus.wait_dns_entries.value,
            register_allowed_ts=datetime.utcnow() + timedelta(minutes=2)
        ).apply()
        return {'result': 'success'}

    is_success = False
    try:

        register_resp = await regru_client.register(
            domain_name=domain,
            nss=config.regru_config['nss'],
            enduser_ip=ip,
            contacts=config.regru_config['contacts']
        )

        is_success = register_resp['result'] == 'success'
        assert is_success, register_resp

    except Exception as error:
        error_log.error("register error of %s@%s for %s. Error: %s", login, domain, uid, str(error), exc_info=True)

    passport = await get_passport_client()
    if is_success:
        # if the request fails here
        # worker could repair this situation
        # by observing the pending registrar statuses

        await domain_record.update(
            status=MailUserDomainStatus.wait_dns_entries.value,
            register_allowed_ts=get_now() + timedelta(days=180),
            service_id=str(register_resp['answer']['service_id']),
        ).apply()
        try:
            await passport.domain_add(domain, config.mail_domains_shared_admin_uid)
        except Exception as error:
            # it's ok if domain is already added to admin uid
            default_log.info('domain %s already added to uid %s. Error %s', domain, config.mail_domains_shared_admin_uid, error)

        return {
            'result': 'success'
        }
    # couldn't register domain
    # save state to database

    await domain_record.update(
        status=MailUserDomainStatus.failed.value,
        register_allowed_ts=get_now()
    ).apply()
    try:
        await passport.delete_pdd_alias(uid)
    except Exception:
        default_log.info(
            f'could not delete alias of {uid}, {domain} when cancelling domain')

    regru_error_code = register_resp.get('error_code')

    if regru_error_code in REGRU_ERRORS_BAD_INPUT:
        raise RegruException(status_code=400, code=regru_error_code)

    if regru_error_code in REGRU_ERRORS_UNAVAILABLE:
        raise RegruException(code=regru_error_code)

    raise RegruException()


@router.post('/cancel_subscription/{uid}')
async def domain_cancel(uid: str, domain: str = None):
    res = await domain_cancel_internal(uid, domain, admin=False)
    return res


async def domain_cancel_internal(uid: str, domain: str = None, admin=False):
    wheres = [
        MailUserDomain.uid == uid,
        MailUserDomain.status.in_(ACTIVE_MAILUSER_DOMAIN_STATUSES_STR)
    ]

    if domain:
        wheres.append(MailUserDomain.domain == domain)

    # only one domain in ACTIVE_MAILUSER_DOMAIN_STATUSES_STR is allowed
    record = await MailUserDomain.query.where(
        and_(*wheres)
    ).gino.first()

    new_status = MailUserDomainStatus.cancelled_by_user.value
    if admin:
        new_status = MailUserDomainStatus.cancelled_by_admin.value

    if record:
        await record.update(status=new_status).apply()

    passport = await get_passport_client()
    try:
        await passport.delete_pdd_alias(uid)
    except Exception:
        log.debug(
            'could not delete alias of %s, when cancelling domain %s', uid, domain)

    try:
        await send_cancelled_domain_mail(to_yandex_puid=uid, email=f'{record.login}@{record.domain}')
    except Exception:
        log.debug(
            'could not send email to %s, when cancelling domain %s', uid, domain)

    if admin:
        default_log.info(f'User {record.uid} remove beautiful address {record.login}@{record.domain} by admin')
    else:
        default_log.info(f'User {record.uid} remove beautiful address {record.login}@{record.domain} by user')

    return {
        'result': 'success'
    }


@router.get('/status/{uid}')
async def domain_status(uid: str, domain: str = None):
    wheres = [
        MailUserDomain.uid == uid,
    ]

    if domain:
        wheres.append(MailUserDomain.domain == domain)

    wheres_active = wheres + \
        [MailUserDomain.status.in_(ACTIVE_MAILUSER_DOMAIN_STATUSES_STR)]

    # try to find active domain
    record = await MailUserDomain.query.where(
        and_(*(wheres_active))
    ).gino.first()

    if record is None:
        wheres_inactive = wheres + \
            [~MailUserDomain.status.in_(ACTIVE_MAILUSER_DOMAIN_STATUSES_STR)]
        record = await MailUserDomain.query.where(
            and_(*(wheres_inactive))
        ).order_by(desc(MailUserDomain.updated_at)).gino.first()

    if record is None:
        raise HTTPException(
            status.HTTP_404_NOT_FOUND,
            'domain not found'
        )

    register_allowed = record.status not in ACTIVE_MAILUSER_DOMAIN_STATUSES_STR and record.register_allowed_ts < get_now()

    return {
        'status': record.status,
        'domain': record.domain,
        'login': record.login,
        'register_allowed': register_allowed,
        'register_allowed_ts': record.register_allowed_ts
    }


@router.get("/magic/info")
async def magic_info(uid: str):
    # try to find active domain
    records = await MailUserDomain.query.where(
        MailUserDomain.uid == uid
    ).gino.all()

    if records is None:
        return []

    return [
        {
            'login': record.login,
            'domain': record.domain,
            'status': record.status,
            'register_date': record.updated_at,
            'status_update_date': record.updated_at
        }
        for record in records
    ]


@router.post("/magic/cancel")
async def magic_cancel(uid: str, domain=None):
    res = await domain_cancel_internal(uid, domain, admin=True)
    return res


@router.post("/change_login/{uid}")
async def change_login(uid: str, login: str):
    check_login_allowed_or_throw(login)
    record = await MailUserDomain.query.where(
        and_(MailUserDomain.uid == uid,
             MailUserDomain.status == MailUserDomainStatus.registered
             )
    ).gino.first()
    if record is None:
        raise HTTPException(
            status.HTTP_404_NOT_FOUND,
            'registered domain not found',
        )

    passport = await get_passport_client()

    prev_pdd_alias = f'{record.login}@{record.domain}'
    pdd_alias = f'{login}@{record.domain}'

    async def revert_changes():
        try:
            await passport.add_pdd_alias(uid, prev_pdd_alias)
        except Exception:
            log.error(f'could not revert alias {prev_pdd_alias} of {uid}')

    try:
        await passport.delete_pdd_alias(uid)
    except Exception:
        log.debug(f'could not delete alias of {uid} when changing login from {prev_pdd_alias}')
    try:
        await passport.add_pdd_alias(uid, pdd_alias)
    except Exception:
        log.debug(f'could not add alias of {uid}, when changing login to {pdd_alias}')
        await revert_changes()

    mailsettings_client = await get_mailsettings_client()
    try:
        await mailsettings_client.update_profile(record.uid, {
            'default_email': pdd_alias
        })
    except Exception:
        log.debug(f'could not update default email for uid {uid} and alias {pdd_alias}')

    await record.update(login=login).apply()
    return {
        'result': 'success'
    }


#  CHEMODAN-79141 will be removed after testing
@router.post("/change_register_allowed_ts/{uid}")
async def change_register_allowed_ts(uid: str, domain: str):
    if uid not in config.users_allowed_to_set_register_ts:
        raise HTTPException(
            status.HTTP_422_UNPROCESSABLE_ENTITY,
            'User not in allowed list',
        )
    domains = await MailUserDomain.query.where(and_(
        MailUserDomain.uid == uid,
        MailUserDomain.domain == domain
    )).gino.all()
    for domain in domains:
        await domain.update(register_allowed_ts=datetime.utcnow()).apply()
    return {
        'result': 'success'
    }
