from binascii import Error as BinasciiError

from fastapi import APIRouter, status
from starlette.exceptions import HTTPException
from starlette.responses import Response
from sqlalchemy import and_

from intranet.domenator.src.db import DomainToken
from ..schemas.domain_token import (
    DataByTokenSchema,
    DomainTokenOutSchema,
)
from ...logic.cryptography import serialize_token, unserialize_token, generate_token
from ...logic.types import PDDVersion

router = APIRouter()


@router.get(
    "/{token}/{pdd_version}/",
    response_model=DataByTokenSchema,
    response_model_by_alias=False,
)
async def data_by_token(token: str, pdd_version: PDDVersion):
    try:
        serialized_token = serialize_token(pdd_version, token)
    except (TypeError, BinasciiError):
        raise HTTPException(
            status.HTTP_404_NOT_FOUND,
            "{} is not found".format(DomainToken.__name__),
        )

    return await DomainToken.query.where(
        and_(
            DomainToken.pdd_version == pdd_version,
            DomainToken.token == serialized_token,
        )
    ).gino.first_or_404()


@router.post(
    "/{uid}/{domain}/{pdd_version}/",
    response_model=DomainTokenOutSchema,
    status_code=status.HTTP_201_CREATED,
)
async def create_token(uid: str, domain: str, pdd_version: PDDVersion):
    existing_token = await DomainToken.query.where(
        and_(
            DomainToken.pdd_version == pdd_version,
            DomainToken.admin_id == uid,
            DomainToken.domain == domain,
        )
    ).gino.first()
    if existing_token:
        token = unserialize_token(
            pdd_version=pdd_version,
            token=existing_token.token
        )
    else:
        token = generate_token(
            pdd_version,
            domain,
            uid,
        )
        await DomainToken.create(
            pdd_version=pdd_version,
            admin_id=uid,
            token=serialize_token(
                pdd_version=pdd_version,
                token=token,
            ),
            domain=domain,
        )

    return {'token': token}


@router.delete(
    "/{uid}/{domain}/{pdd_version}/",
    status_code=status.HTTP_204_NO_CONTENT,
    response_class=Response,
)
async def delete_token(uid: str, domain: str, pdd_version: PDDVersion):
    token_obj = await DomainToken.query.where(
        and_(
            DomainToken.pdd_version == pdd_version,
            DomainToken.admin_id == uid,
            DomainToken.domain == domain,
        )
    ).gino.first_or_404()
    await token_obj.delete()
