import logging
from fastapi import APIRouter, Depends, status
from sqlalchemy import and_
from starlette.exceptions import HTTPException
from starlette.responses import Response
from typing import Set, Optional, List

from intranet.domenator.src.api.common import ListParser
from intranet.domenator.src.api.schemas.private.domain import (
    PrivateDomainOutSchema,
    PrivateDomainUpdateSchema,
)
from intranet.domenator.src.db.domain.models import Domain
from intranet.domenator.src.logic.blackbox import get_blackbox_client
from intranet.domenator.src.logic.passport import get_passport_client
from intranet.domenator.src.logic.domain import to_punycode
from intranet.domenator.src.logic.passport import delete_domain_in_passport
from intranet.domenator.src.logic.event import save_event
from intranet.domenator.src.logic.event.events import DomainChangedEvent, DomainMasterChangedEvent

log = logging.getLogger(__name__)
router = APIRouter()


@router.get("/", response_model=List[PrivateDomainOutSchema], response_model_by_alias=False)
async def get(
    org_id: Set[str] = Depends(ListParser('org_id', container=set)),
    name: Optional[str] = None,
    master: Optional[bool] = None,
    owned: Optional[bool] = None,
):
    if org_id is None and name is None:
        raise HTTPException(
            status.HTTP_422_UNPROCESSABLE_ENTITY,
            'Specify org_id or name.',
        )

    wheres = []
    if org_id is not None:
        wheres.append(Domain.org_id.in_(org_id))
    if name is not None:
        wheres.append(Domain.name == to_punycode(name))
    if master is not None:
        wheres.append(Domain.master == bool(master))
    if owned is not None:
        wheres.append(Domain.owned == bool(owned))

    return await Domain.query.where(and_(*wheres)).gino.all()


@router.patch("/", response_class=Response)
async def patch(
    org_id: str,
    data: PrivateDomainUpdateSchema,
    name: Optional[str] = None,
    owned: Optional[bool] = None,
    master: Optional[bool] = None,
):
    blackbox = await get_blackbox_client()
    passport = await get_passport_client()
    set_master = data.master
    if set_master is not None and name is None:
        raise HTTPException(
            status.HTTP_400_BAD_REQUEST,
            'You should provide domain name when making it master for the organization'
        )
    if set_master is False:
        raise HTTPException(
            status.HTTP_400_BAD_REQUEST,
            f'You can make domain "{name}" master, but you cannot unmake it master'
        )

    if set_master:
        domain_to_become_master = await Domain.query.where(
            and_(Domain.org_id == org_id, Domain.name == name)
        ).gino.first()
        if not domain_to_become_master.owned:
            raise HTTPException(
                status.HTTP_422_UNPROCESSABLE_ENTITY,
                f'Cannot make master not owned domain "{name}"'
            )
        if domain_to_become_master.blocked_at:
            raise HTTPException(
                status.HTTP_422_UNPROCESSABLE_ENTITY,
                f'Cannot make master blocked domain "{name}"'
            )

        current_master = await Domain.query.where(
            and_(Domain.org_id == org_id, Domain.master == True)  # noqa
        ).gino.first()
        if not current_master:
            raise HTTPException(
                status.HTTP_422_UNPROCESSABLE_ENTITY,
                'There is no master domain for this organization'
            )

        if current_master.name != domain_to_become_master.name:
            old_master_bb_info = await blackbox.get_domain_info(current_master.name)
            new_master_bb_info = await blackbox.get_domain_info(name)

            log.info('Saving event about changing master from %s to %s for org_id %s',
                     name, current_master.name, org_id)
            await save_event(DomainMasterChangedEvent(
                domain=name,
                org_id=org_id,
                admin_uid=domain_to_become_master.admin_id,
                old_master_domain=current_master.name,
                old_master_admin_uid=current_master.admin_id,
            ))
            log.info(
                'Changing current master in passport for org_id %s from %s to %s (from domid %s to domid %s)',
                org_id, name, current_master.name, old_master_bb_info['domain_id'], new_master_bb_info['domain_id']
            )
            await passport.set_master_domain(old_master_bb_info['domain_id'], new_master_bb_info['domain_id'])

    data_for_patch = data.dict(exclude_unset=True)

    wheres = [Domain.org_id == org_id]
    if name is not None:
        wheres.append(Domain.name == to_punycode(name))
    if owned is not None:
        wheres.append(Domain.owned == owned)
    if master is not None:
        wheres.append(Domain.master == master)

    domains = await Domain.query.where(and_(*wheres)).gino.all()
    for domain in domains:
        old_data = {
            'owned': domain.owned,
            'master': domain.master,
            'display': domain.display,
            'blocked_at': domain.blocked_at,
        }
        new_data = old_data.copy()
        new_data.update(data_for_patch)
        await save_event(DomainChangedEvent(
            domain=domain.name,
            org_id=org_id,
            old_data=old_data,
            new_data=new_data,
        ))

    if set_master is not None:
        domains = await Domain.query.where(
            and_(
                Domain.org_id == org_id,
                Domain.name != name,
                Domain.master == True,  # noqa
            )
        ).gino.all()
        for domain in domains:
            old_data = {
                'owned': domain.owned,
                'master': domain.master,
                'display': domain.display,
                'blocked_at': domain.blocked_at,
            }
            new_data = old_data.copy()
            new_data['master'] = False
            await save_event(DomainChangedEvent(
                domain=domain.name,
                org_id=org_id,
                old_data=old_data,
                new_data=new_data,
            ))

        # we should reset master for every other domain of that organization
        await Domain.update.values(master=False).where(
            and_(Domain.org_id == org_id, Domain.name != name)
        ).gino.status()

    await Domain.update.values(**data_for_patch).where(and_(*wheres)).gino.status()


@router.delete("/delete-all/{org_id}", response_class=Response)
async def delete_all(org_id: str):
    master: Domain = Domain.query.where(and_(
        Domain.org_id == org_id,
        Domain.master == True,  # noqa
    )).gino.first()

    aliases: List[Domain] = Domain.query.where(and_(
        Domain.org_id == org_id,
        Domain.master == False,  # noqa
        owned == True,  # noqa
    )).gino.all()

    if master:
        bb = await get_blackbox_client()
        master_domain_bb_info = await bb.get_domain_info(master.name)
        for alias in aliases:
            await delete_domain_in_passport(
                domain_name=alias.name,
                admin_uid=alias.admin_id,
                master_domain_id=master_domain_bb_info['domain_id'],
            )
        await delete_domain_in_passport(
            domain_name=master.name,
            admin_uid=master.admin_id
        )

    await Domain.delete.where(Domain.org_id == org_id).gino.status()
