from fastapi import APIRouter, status
from sqlalchemy import and_
from starlette.exceptions import HTTPException
from starlette.responses import Response

from intranet.domenator.src.db.registrar.models import Registrar
from ..schemas.registrar import (
    RegistrarOutSchema,
    RegistrarUpdateSchema,
    RegistrarByTokenOutSchema,
    RegistrarTokenOutSchema,
)
from ...logic.cryptography import serialize_token, unserialize_token, generate_token
from ...logic.registrar.password import encrypt_registrar_password
from ...logic.types import PDDVersion

router = APIRouter()


@router.get(
    '/{registrar_id}/',
    response_model=RegistrarOutSchema,
    response_model_by_alias=False,
)
async def get_registrar(registrar_id: str):
    is_pdd_id = False
    pdd_id = None
    pdd_version = None
    try:
        registrar_id = int(registrar_id)
    except ValueError:
        try:
            pdd_version, pdd_id = registrar_id.split(':')
        except ValueError:
            raise HTTPException(
                status.HTTP_422_UNPROCESSABLE_ENTITY,
                'Invalid registrar id passed',
            )
        is_pdd_id = True

    if is_pdd_id:
        registrar = await Registrar.query.where(
            and_(
                Registrar.pdd_id == int(pdd_id),
                Registrar.pdd_version == pdd_version,
            )
        ).gino.first_or_404()
    else:
        registrar = await Registrar.get_or_404(registrar_id)

    return registrar


@router.patch(
    '/{registrar_id}/',
    response_class=Response,
)
async def patch_registrar(registrar_id: int, item: RegistrarUpdateSchema):
    registrar = await Registrar.get_or_404(registrar_id)
    data_for_patch = item.dict(exclude_unset=True)

    if 'password' in data_for_patch:
        password, iv, plength = encrypt_registrar_password(registrar.pdd_version, data_for_patch['password'])
        data_for_patch['password'] = password
        data_for_patch['iv'] = iv
        data_for_patch['plength'] = plength

    await registrar.update(**data_for_patch).apply()


@router.get(
    '/token/v2/{token}/',
    response_model=RegistrarByTokenOutSchema,
    response_model_by_alias=False,
)
async def get_registrar_by_token(token: str):
    serialized_token = serialize_token(PDDVersion.new, token)
    return await Registrar.query.where(Registrar.token == serialized_token).gino.first_or_404()


@router.get(
    '/uid/v2/{uid}/',
    response_model=RegistrarOutSchema,
    response_model_by_alias=False,
)
async def get_registrar_by_uid(uid: str):
    return await Registrar.query.where(
        and_(
            Registrar.admin_id == uid,
            Registrar.pdd_version == PDDVersion.new,
        )
    ).gino.first_or_404()


@router.post(
    '/{registrar_id}/token/',
    response_model=RegistrarTokenOutSchema,
    response_model_by_alias=False,
)
async def get_or_create_registrar_token(registrar_id: int):
    registrar = await Registrar.get_or_404(registrar_id)

    if registrar.pdd_version == PDDVersion.old:
        raise HTTPException(
            status.HTTP_422_UNPROCESSABLE_ENTITY,
            'Tokens are not available for registrars with old pdd version',
        )

    if registrar.token:
        raw_token = unserialize_token(registrar.pdd_version, registrar.token)
    else:
        raw_token = generate_token(registrar.pdd_version, registrar.id)
        serialized_token = serialize_token(registrar.pdd_version, raw_token)
        await registrar.update(token=serialized_token).apply()

    return {'token': raw_token}


@router.delete(
    '/{registrar_id}/token/',
    response_class=Response,
)
async def delete_registrar_token(registrar_id: int):
    registrar = await Registrar.get_or_404(registrar_id)
    await registrar.update(token=None).apply()
