import logging
from logging.config import dictConfig

from asgi_yauth.middleware import (
    YauthTestMiddleware,
    YauthMiddleware,
)
from fastapi import FastAPI, APIRouter

from intranet.domenator.src.api.routes.domain import router as domain_router
from intranet.domenator.src.api.routes.domain_token import router as domain_token_router
from intranet.domenator.src.api.routes.private.domain import router as private_domain_router
from intranet.domenator.src.api.routes.registrar import router as registrar_router
from intranet.domenator.src.common_routes import router as common_router
from intranet.domenator.src.db import db
from intranet.domenator.src.settings import config
from intranet.domenator.src.domenator_logging.settings import LOGGING
from .logic.exceptions.base import BaseDomenatorException
from .logic.exceptions.exception_handler import base_exception_handler, unhandled_exception_handler
from .middleware import yauth_require
from intranet.domenator.src.domenator_logging.middleware import log_context_middleware
from intranet.domenator.src.utils.tasks import init_tasks
dictConfig(LOGGING)

logger = logging.getLogger(__name__)


def get_api_router():
    api_router = APIRouter()

    api_router.include_router(
        domain_router,
        prefix="/domains",
        tags=["domains"],
    )

    api_router.include_router(
        domain_token_router,
        prefix="/domain-token",
        tags=["domain-token"],
    )

    api_router.include_router(
        registrar_router,
        prefix='/registrar',
        tags=['registrar'],
    )

    api_router.include_router(
        private_domain_router,
        prefix='/private/domains',
        tags=['private'],
    )

    api_router.include_router(
        common_router,
        prefix='/common',
        tags=['common'],
    )

    return api_router


def add_middlewares(app):
    """
    Тут важен порядок, middleware добавленые
    в начале этого метода будут вызваны после тех
    что идут в конце
    """
    app.middleware("http")(log_context_middleware)
    app.middleware("http")(yauth_require)

    if config.env_type == 'development':
        yauth_middleware = YauthTestMiddleware
    else:
        yauth_middleware = YauthMiddleware
    app.add_middleware(yauth_middleware, config=config)


def add_exception_handlers(app):
    app.exception_handler(BaseDomenatorException)(base_exception_handler)
    app.exception_handler(Exception)(unhandled_exception_handler)


def get_app():
    app = FastAPI(
        title='domenator',
        openapi_url=f'{config.docs_prefix}/openapi.json',
        docs_url=f'{config.docs_prefix}/swagger',
        redoc_url=f'{config.docs_prefix}/redoc',
    )

    router = get_api_router()

    app.include_router(
        router,
        prefix="/api",
    )

    db.init_app(app)
    init_tasks(app)
    add_exception_handlers(app)
    add_middlewares(app)
    return app


app = get_app()
