from cached_property import cached_property
from ylog.context import put_to_context, pop_from_context

from .base import BaseAsyncContext
from .collector import ContextCollector
from intranet.domenator.src.domenator_logging.logger import get_logger

access_log = get_logger(log_name='dom_access')
error_log = get_logger(log_name='dom_error')
default_log = get_logger(log_name='dom_default')


class HttpLogContext(BaseAsyncContext):
    TAG = 'http'

    def __init__(self, request, **kwargs):
        self.request = request
        super().__init__(**kwargs)

    def _before_exit(self):
        super()._before_exit()
        pop_from_context('response')

    def log_after_execution(self, context):
        user = context.get('user') or {}
        request = context.get('request') or {}
        response = context.get('response') or {}

        access_log.info(
            'http request %s: %s %.3fs',
            request.get('method'),
            request.get('uri'),
            context.get('request_time'),
            extra={
                'proto': request.get('proto'),
                'method': request.get('method'),
                'uri': request.get('uri'),
                'uid': user.get('uid'),
                'headers': request.get('headers'),
                'status': response.get('status_code') or 500,
                'request_time': context.get('request_time'),
            }
        )

    @cached_property
    def collector(self):
        return ContextCollector(request=self.request)


http_log_context = HttpLogContext


async def log_context_middleware(request, call_next):
    with http_log_context(request):
        try:
            response = await call_next(request)
            put_to_context('response', {
                'status_code': response.status_code
            })
        except Exception as exc:
            error_log.error('Unhandled exception: %s', repr(exc))
            default_log.exception(str(exc))
            raise
    return response
