import time
import urllib.parse
import logging
import asyncio
import aiohttp
from typing import Optional, Any

from async_clients.clients.base import BaseClient
from ylog.context import log_context
from intranet.domenator.src.domenator_logging.logger import get_logger

from tenacity import (
    AsyncRetrying,
    RetryError,
    stop_after_attempt,
    wait_exponential,
    retry_if_exception_type,
    after_log,
    TryAgain,
)

from async_clients.exceptions.base import (
    NoRetriesLeft,
    AIOHTTPClientException,
)

requests_log = get_logger(log_name='dom_requests')
error_log = get_logger(log_name='dom_error')
default_log = get_logger(log_name='dom_default')

HTTP_200 = 200


class ProfilingClientMixin(BaseClient):
    CLIENT_NAME = None

    async def _make_request(
        self,
        path: str,
        method: str = 'get',
        params: Optional[dict] = None,
        json: Optional[dict] = None,
        data: Optional[Any] = None,
        headers: Optional[dict] = None,
        timeout: int = None,
        response_type: str = None,
        **kwargs
    ):
        url = urllib.parse.urljoin(self.host, path)

        params = self._prepare_params(params)
        retries = self.get_retries(
            path,
            method,
            params,
            json,
            data,
            headers,
            timeout,
            **kwargs
        )
        proxy, proxy_headers = self.get_proxy_params()

        async with self.get_session(timeout=timeout) as session:
            try:
                async for attempt in AsyncRetrying(
                    stop=stop_after_attempt(retries),
                    retry=retry_if_exception_type(TryAgain),
                    wait=wait_exponential(multiplier=self.wait_multiplier),
                    after=after_log(default_log, logging.WARNING)
                ):
                    with attempt:
                        try:
                            start_time = time.time()
                            async with getattr(session, method)(
                                url=url,
                                params=params,
                                headers=headers,
                                json=json,
                                data=data,
                                proxy=proxy,
                                proxy_headers=proxy_headers,
                                ssl=not self.USE_ZORA
                            ) as response:
                                request_time = time.time() - start_time

                                context_data = {
                                    'client': self.CLIENT_NAME,
                                    'url': url,
                                    'method': method.upper(),
                                    'request_time': request_time,
                                }
                                with log_context(upstream=context_data):
                                    requests_log.info(
                                        '%s %s: %s (attempt: %s) took %.3fs',
                                        method.upper(),
                                        url,
                                        response.status,
                                        attempt.retry_state.attempt_number,
                                        request_time,
                                        extra={
                                            'request_time': request_time,
                                            'client_name': self.CLIENT_NAME,
                                        }
                                    )

                                return await self.parse_response(
                                    response=response,
                                    response_type=response_type,
                                    **kwargs
                                )
                        except (
                            aiohttp.ServerTimeoutError,
                            aiohttp.ServerDisconnectedError,
                            asyncio.TimeoutError,
                        ) as exc:
                            default_log.warning(
                                '%s %s raised "%s". Retrying request.', method, url, str(exc))
                            raise TryAgain()
                        except aiohttp.ClientError as exc:
                            headers = getattr(exc, 'headers', None)
                            if headers is not None:
                                exc.headers = self._prepare_headers_for_logging(
                                    headers)
                            message = f'Got aiohttp exception: {repr(exc)}'
                            raise AIOHTTPClientException(
                                message, original_exception=exc)
            except RetryError:
                raise NoRetriesLeft()
