import logging
import time

from datetime import datetime
from fastapi import status

from sqlalchemy import and_, or_
from starlette.exceptions import HTTPException
from typing import Optional, Tuple
from async_clients.exceptions.passport import DomainAlreadyExists

from intranet.domenator.src.settings import config
from intranet.domenator.src.logic.blackbox import get_blackbox_client
from intranet.domenator.src.logic.connect import get_connect_client
from intranet.domenator.src.logic.event import save_event
from intranet.domenator.src.logic.event.events import DomainOccupiedEvent, DomainAddedEvent
from intranet.domenator.src.logic.exceptions.domain import InvalidDomainException, DuplicateDomainException
from intranet.domenator.src.logic.fouras import get_fouras_client
from intranet.domenator.src.logic.gendarme import get_gendarme_client
from intranet.domenator.src.logic.passport import get_passport_client
from intranet.domenator.src.logic.webmaster import get_webmaster_client
from intranet.domenator.src.db.domain.models import Domain, DomainAction, DomainHistory

log = logging.getLogger(__name__)


def is_tech(domain_name: str) -> bool:
    return domain_name.endswith(config.tech_domain_part)


async def save_domain_history(domain: Domain, action: DomainAction, author_id: Optional[str] = None):
    await DomainHistory.create(
        org_id=domain.org_id,
        name=domain.name,
        action=action,
        author_id=author_id,
    )


async def sync_domain_state(org_id: str, domain_name: str, admin_uid: str, registrar_id: Optional[int] = None):
    log.debug('sync domain')

    if domain_name in config.yandex_team_org_domains:
        return

    domain: Domain = await Domain.query.where(
        and_(
            Domain.name == domain_name,
            Domain.org_id == org_id,
        )
    ).gino.first()

    # возможно домена уже нет в организации
    if not domain or domain.owned:
        return

    webmaster = await get_webmaster_client()
    owned_in_webmaster = await webmaster.is_verified(domain_name, admin_uid)
    if owned_in_webmaster:
        old_owner_org_id, old_owner_new_master_name, old_owner_new_master_tech = await disable_domain(
            domain_name, org_id, admin_uid)

        # проставляем в текущей
        is_master = await create_domain_in_passport(org_id, domain_name, admin_uid)
        await domain.update(
            owned=True,
            validated_at=datetime.utcnow(),
        ).apply()

        if is_master:
            await after_first_domain_became_confirmed(org_id, domain, admin_uid)

        await save_domain_history(domain, DomainAction.domain_occupied)
        await save_event(DomainOccupiedEvent(
            domain=domain.name,
            new_owner_org_id=org_id,
            new_owner_new_domain_is_master=is_master,
            old_owner_org_id=old_owner_org_id,
            old_owner_new_master_name=old_owner_new_master_name,
            old_owner_new_master_tech=old_owner_new_master_tech,
        ))

        connect = await get_connect_client()
        await connect.notify_about_domain_occupied(
            domain=domain.name,
            new_owner_org_id=int(org_id),
            new_owner_new_domain_is_master=is_master,
            old_owner_org_id=old_owner_org_id,
            old_owner_new_master_name=old_owner_new_master_name,
            old_owner_new_master_tech=old_owner_new_master_tech,
            registrar_id=registrar_id,
        )
    else:
        response = await webmaster.info(domain_name, admin_uid, ignore_errors='USER__HOST_NOT_ADDED')
        data = response['data'] or {}
        verification_type = data.get('verificationType')
        verification_status = data.get('verificationStatus')
        # Если это домен от регистратора, то надо запустить проверку заново
        if verification_type == 'PDD_EMU' and verification_status == 'VERIFICATION_FAILED':
            await webmaster.verify(domain_name, admin_uid, verification_type)


async def create_domain_in_passport(org_id: str, domain_name: str, admin_uid: str):
    master_domain: Domain = await Domain.query.where(
        and_(
            Domain.org_id == org_id,
            Domain.master == True,  # noqa
        )
    ).gino.first()

    add_master = False
    # если мастер домена нет или мы пытаемся его подтвердить, то будем добавлять домен как основной
    if not master_domain or master_domain.name == domain_name:
        add_master = True

    passport = await get_passport_client()
    if add_master:
        await passport.domain_add(domain_name, admin_uid)
    else:
        blackbox = await get_blackbox_client()
        master_blackbox_info = await blackbox.get_domain_info(master_domain.name)
        await passport.domain_alias_add(master_blackbox_info['domain_id'], domain_name)
        alias_blackbox_info = await blackbox.get_domain_info(domain_name)
        await passport.domain_edit(alias_blackbox_info['domain_id'], {'admin_uid': admin_uid})

    gendarme = await get_gendarme_client()
    await gendarme.recheck(domain_name)
    fouras = await get_fouras_client()
    await fouras.get_or_gen_domain_key(domain_name)

    return add_master


def generate_tech_domain(domain_name: str, org_id: str):
    # генерирует технический домен для организации <domain_name>-<org_id>.yaconnect.com
    # в имени точки заменяет на дефис
    return '{domain_name}-{org_id}{domain_part}'.format(
        domain_name=domain_name.replace('.', '-'),
        org_id=org_id,
        domain_part=config.tech_domain_part
    )


async def disable_domain(domain_name: str, excluded_org_id: str, new_owner_uid: str) -> Tuple[Optional[int], Optional[str], Optional[bool]]:
    # Открепляем домен от организации, где он подтвержден, кроме excluded_org_id
    # если он был мастером, то делаем другой подтвержденный мастером или, если его нет, то генерируем технический.

    domain: Domain = await Domain.query.where(
        and_(
            Domain.name == domain_name,
            Domain.owned == True,  # noqa
            Domain.org_id != excluded_org_id,
        )
    ).gino.first()

    if not domain:
        return None, None, None

    if is_portal_domain(domain_name):
        raise HTTPException(
            status.HTTP_422_UNPROCESSABLE_ENTITY,
            'Can not disable portal domain',
        )

    new_master_domain_name = None
    tech = False

    blackbox = await get_blackbox_client()
    passport = await get_passport_client()

    if domain.master:
        other_owned_domain = await Domain.query.where(and_(
            Domain.org_id == domain.org_id,
            Domain.owned == True,  # noqa
            Domain.name != domain.name,
        )).gino.first()
        if other_owned_domain:
            new_master_domain_name = other_owned_domain['name']
        else:
            new_master_domain_name = generate_tech_domain(
                domain_name, domain.org_id)
            tech = True
            new_master_domain: Domain = await Domain.create(
                org_id=domain.org_id,
                name=new_master_domain_name,
                owned=True,
                admin_id=domain.admin_id,
                master=True,
            )
            await save_domain_history(new_master_domain, DomainAction.domain_added)
            await save_event(DomainAddedEvent(
                domain=domain.name,
                org_id=domain.org_id,
                author_id=None,
            ))

            await create_domain_in_passport(domain.org_id, new_master_domain_name, domain.admin_id)
            time.sleep(5)  # даем время репликам паспорта

        old_master_bb_info = await blackbox.get_domain_info(domain.name)
        new_master_bb_info = await blackbox.get_domain_info(new_master_domain_name)

        await passport.set_master_domain(old_master_bb_info['domain_id'], new_master_bb_info['domain_id'])
        time.sleep(5)  # даем время репликам паспорта
        await passport.domain_alias_delete(
            new_master_bb_info['domain_id'],
            old_master_bb_info['domain_id']
        )
    else:
        alias_bb_info = await blackbox.get_domain_info(domain.name)
        master_domain_name = alias_bb_info['master_domain']
        master_bb_info = await blackbox.get_domain_info(master_domain_name)
        await passport.domain_alias_delete(master_bb_info['domain_id'], alias_bb_info['domain_id'])

    await domain.delete()
    await save_domain_history(domain, DomainAction.domain_auto_handover)

    # запускаем перепроверку для старого владельца, чтобы оторвать у него права в вебмастере
    webmaster = await get_webmaster_client()
    await webmaster.reset(domain.name, domain.admin_id, new_owner_uid)

    return domain.org_id, new_master_domain_name, tech


async def after_first_domain_became_confirmed(org_id: str, domain: Domain, admin_uid: str):
    await domain.update(master=True).apply()

    blackbox = await get_blackbox_client()
    domain_blackbox_info = await blackbox.get_domain_info(
        domain_name=domain.name,
        admin_uid=admin_uid,
    )

    organization_name = 'org_id:{0}'.format(org_id)
    passport = await get_passport_client()
    await passport.domain_edit(domain_blackbox_info['domain_id'], {
        'organization_name': organization_name,
    })


def to_punycode(text):
    try:
        return text.encode('idna').decode()
    except UnicodeError as e:
        if 'too long' in str(e):
            message = 'Domain name {0} too long'.format(text)
            raise ValueError(message)
        raise


def from_punycode(text):
    return text.encode().decode('idna')


def is_portal_domain(domain_name: str) -> bool:
    portal_domains = {
        'infohelp.ooo',
        'reclamat.ru',
        'finam.ru',
        'index.ua',
        'pravmail.ru',
        'go-go.by',
        '211.ru',
        'bitrix24.com',
        'spark-mail.ru',
        'mymeil.ru',
        'akado-ural.ru',
        'setitagila.ru',
        '360.by',
        'mail.izh.com',
        'tmpk.ru',
        'kazan.ru',
        'stud.etu.ru',
        'xn--80aqa9a.xn--p1ai',
        'kuz.ru',
        'mellior.ru',
        'nur.kz',
        'citydom.ru',
        'fxmail.ru',
        'ok100.shop',
        'cn.ru',
        'udm.ru',
        'hardor.ru',
        'rostov.ru',
        '60.ru',
        '35.ru',
        'triya.ru',
        'spartak.ru',
        'bc.su',
        'bgnsk.ru',
        'rtural.ru',
        'bitrix24.es',
        'orenburgdom.ru',
        'mabila.ua',
        'instarlogistics.com',
        'forum.nn.ru',
        'terminal.ru',
        'udmlink.ru',
        'yahbox.ru',
        'infsx.ru',
        'omsk1.ru',
        'samaradom.ru',
        'drummer.ru',
        'brs.ua',
        'olam.uz',
        'weburg.me',
        '59.ru',
        '52.ru',
        'wapalta.mobi',
        'medlab.space',
        'umi.ru',
        '74.ru',
        'bitrix24.ru',
        'uacity.com',
        '191.su',
        'stagila.ru',
        't-sk.ru',
        'rx24.ru',
        'squareyards.co.in',
        'n1.by',
        '162.by',
        'feo.ua',
        'adv-mail1.ru',
        'agent.sgmsk.ru',
        'universitys.ru',
        'ah.org.ua',
        'student.vags.ru',
        's125.ru',
        'date.by',
        'hardcore.ru',
        'aucu.ru',
        'yugs.ru',
        'tambo.ru',
        'ksmmmo.org.tr',
        'idz.ru',
        'jcatmail.ru',
        'dmitry.by',
        'uainet.net',
        'otvcv.ru',
        'nav.uz',
        'sterlitamak1.ru',
        '178.ru',
        'e-izhevsk.ru',
        'ninodom.ru',
        'penzadom.ru',
        'chelskdom.ru',
        'kazandom.ru',
        'gogo.by',
        'yoladom.ru',
        'tyumendom.ru',
        'for-sage.info',
        'mail.udm.ru',
        'moag.ru',
        'ingushetia.org',
        'e-mail.ru',
        'xn--b1amarfbvj.xn--p1ai',
        'volgodom.ru',
        'fermer.ru',
        'mitino.ru',
        'gallerix.ru',
        'mirttk.ru',
        'xakep.ru',
        'alexandr.by',
        'dom.raid.ru',
        'chebnet.com',
        'nlstar.com',
        'pm.convex.ru',
        'olympus.ru',
        'rsb.ua',
        'xaker.ru',
        '2ch.hk',
        'satka.ru',
        'weburg.ru',
        'hi.ru',
        'goon.ru',
        'psychology-guide.ru',
        'kazakh.ru',
        'permonline.ru',
        'torba.com',
        'sxnarod.com',
        'voliacable.com',
        'nnovgorod.ru',
        'izh.com',
        'bee-life.ru',
        'pushkin.ru',
        'neomail.ru',
        'corpt.ru',
        'status4ka.am',
        '7do.ru',
        'email.su',
        'listler.ru',
        'novo-city.ru',
        'rambbox.ru',
        'kbs.ru',
        'moag.pro',
        'pochta.tvoe.tv',
        'nextmail.ru',
        'fotostrana.ru',
        'rocketdata.io',
        'xn--c1ajbaaoubcdrfia5c.xn--p1ai',
        'ufa1.ru',
        'sergei.by',
        'kp11.ru',
        'pedsovet.su',
        'flylady.su',
        'pusk.by',
        '154.ru',
        'proizhevsk.ru',
        'provoronezh.ru',
        'sochi1.ru',
        'chelfin.ru',
        'byfly.ws',
        'hu2.ru',
        '72.ru',
        'nxt.ru',
        'goldsir.ru',
        'alexey.by',
        'ronl.ru',
        'driver-uber.ru.com',
        'k66.ru',
        'pochta.nn.ru',
        'vlzdom.ru',
        'search.ua',
        'box.ua',
        'chel.ru',
        '68.ru',
        '71.ru',
        'e-mail.ua',
        '2074.ru',
        '66.ru',
        'alpha-mail.ru',
        'epage.ru',
        'lettler.ru',
        'potbox.ru',
        'paso.ru',
        'paso-ru-2208944.yaconnect.com',
        'e1.ru',
        'myttk.ru',
        'rukiizplech.ru',
        'ruki-s.ru',
        'rukis.ru',
        'rukiz.ru',
        'xn--e1adcaij1ad1a1c.xn--p1ai',
        'xn-----olcgdbln9ad7a9c.xn--p1ai',
        'qip.ru',
        'gmxl.ru',
        'maxim.by',
        '29.ru',
        'zavolga.net',
        'volia.ua',
        'belg.ru',
        'programist.ru',
        '14.ru',
        '76.ru',
        'primizt.ru',
        '63.ru',
        'clcorp.ru',
        'mail2000.ru',
        'chetk.info',
        'chita.ru',
        'bitrix24.eu',
        'student.su',
        '43.ru',
        'tolyatty.ru',
        'dezigner.ru',
        'roman.by',
        'scot.land',
        '2010001.ru',
        '2304343.ru',
        'mail.mellior.ru',
        '2304545.ru',
        'cheldiplom.ru',
        'jilfond.ru',
        'tomilino.net',
        'mail2k.ru',
        '45.ru',
        'do-student.ru',
        'bitrix24.in',
        'zhilfond.ru',
        'lombard-avrora.ru',
        'academ.org',
        'dc2b.ru',
        '2ch.so',
        'sochi.com',
        'ruki-iz-plech.ru',
        '116.ru',
        'yenimahalle.bel.tr',
        '404mail.net',
        '161.ru',
        'mail57.ru',
        'sbor.net',
        '26.ru',
        'resurs.kz',
        'v1.ru',
        'minsk.edu.by',
        '56.ru',
        'ormamail.ru',
        'convex.ru',
        'flinfo.ru',
        'chelnydom.ru',
        'bitrix24.kz',
        'omskdom.ru',
        'bitrix24.by',
        'at.ua',
        'apmo.ru',
        'day.az',
        'sport.ru',
        'e-kirov.ru',
        'bitrix24.com.br',
        'skyeng.ru',
        '7405606.ru',
        '7405606.ru.com',
        'century21.ru',
        'cian.century21.ru',
        'tyt.by',
        '86.ru',
        'bitrix24.ua',
        'ohranatruda.ru',
        'tut.by',
        'sknt.ru',
        'bitrix24.de',
        'ekat.ru',
        '48.ru',
        'telemaxdom.ru',
        'autochel.ru',
        '62.ru',
        'domchel.ru',
        '164.ru',
        'chelyabinsk.ru',
        'rcfh.ru',
        'vipmail.ru',
        'mychel.ru',
        '70.ru',
        '93.ru',
        'mgorsk.ru',
        'irkutsk138.ru',
        '89.ru',
        '51.ru',
        '42.ru',
        'mail.imc-i.ru',
        'mailboard.info',
        '53.ru',
        'orbita.co.il',
        'stud.eltech.ru',
        'dynamo.kiev.ua',
        'callofduty.ru',
        'kreditkassa.ru',
        'mail.vegu.ru',
    }
    return to_punycode(domain_name) in portal_domains


async def check_domain_is_correct(org_id: str, domain_name: str, admin_uid: str):
    domain_name = domain_name.lower()
    if '.' not in domain_name:
        raise InvalidDomainException()
    if domain_name in config.domains_blacklist:
        raise InvalidDomainException()
    if is_tech(domain_name):
        raise InvalidDomainException()

    try:
        punycode_name = to_punycode(domain_name)
    except Exception:
        raise InvalidDomainException()

    # проверим, что такого домена нет в этой же организации или в другой организации этого админа
    existent_domain: Optional[Domain] = await Domain.query.where(
        and_(
            Domain.name == punycode_name,
            or_(
                Domain.org_id == org_id,
                Domain.admin_id == admin_uid,
            )
        )
    ).gino.first()
    if existent_domain:
        raise DuplicateDomainException(
            conflicting_org_id=existent_domain.org_id)

    passport = await get_passport_client()
    try:
        await passport.validate_natural_domain(domain_name)
    except DomainAlreadyExists:
        pass
