from typing import Optional

from async_clients.clients.passport import Client as PassportClient
from async_clients.exceptions.base import NoRetriesLeft
from fastapi import status
from starlette.exceptions import HTTPException

from intranet.domenator.src.settings import config
from .blackbox import get_blackbox_client
from .exceptions.common import FailedDependencyException
from .tvm2 import get_service_ticket
from intranet.domenator.src.domenator_logging.middleware.profiling_client import ProfilingClientMixin


class PassportProfilingClient(PassportClient, ProfilingClientMixin):
    CLIENT_NAME = 'passport'


async def get_passport_client() -> PassportProfilingClient:
    return PassportProfilingClient(
        host=config.passport_host,
        consumer_query_param='domenator',
        service_ticket=await get_service_ticket(config.passport_tvm2_client)
    )


async def delete_domain_in_passport(domain_name: str, admin_uid: str, master_domain_id: Optional[int] = None):
    passport_client = await get_passport_client()
    bb_client = await get_blackbox_client()

    is_alias = bool(master_domain_id)

    domain_info = await bb_client.get_domain_info(domain_name)
    if not domain_info:
        return

    if str(domain_info['admin_id']) != admin_uid:
        raise HTTPException(
            status.HTTP_422_UNPROCESSABLE_ENTITY,
            'Wrong admin_id',
        )

    try:
        if is_alias:
            await passport_client.domain_alias_delete(master_domain_id, domain_info['domain_id'])
        else:
            await passport_client.domain_delete(domain_info['domain_id'])
    except NoRetriesLeft:
        raise FailedDependencyException('passport')


async def get_userinfo(uid: str, ip: str):
    bb_client = await get_blackbox_client()
    return await bb_client.get_userinfo(uid, ip)
