import asyncio
from datetime import datetime
from functools import wraps
import logging
from time import time
from typing import Optional
from contextlib import contextmanager

log = logging.getLogger(__name__)


def chunks(lst: list, n: int) -> list:
    """Yield successive n-sized chunks from lst."""
    for i in range(0, len(lst), n):
        yield lst[i:i + n]


def parse_to_datetime(str_datetime: Optional[str] = None) -> Optional[datetime]:
    if str_datetime:
        return datetime.strptime(
            str_datetime,
            '%Y-%m-%dT%H:%M:%S.%f%z'
        )


def timeit(func):
    @contextmanager
    def wrapping_logic(*args, **kwargs):
        start_ts = time()
        yield
        dur = time() - start_ts
        args_for_log = []
        for arg in args:
            if isinstance(arg, (list, tuple)):
                args_for_log.append(arg[:5])
            else:
                args_for_log.append(arg)
        log.debug(f'func:{func.__name__} args: [{args_for_log}, {kwargs}], took: {dur:2.4f} sec')

    @wraps(func)
    def wrapper(*args, **kwargs):
        if not asyncio.iscoroutinefunction(func):
            with wrapping_logic(*args, **kwargs):
                return func(*args, **kwargs)
        else:
            async def tmp():
                with wrapping_logic(*args, **kwargs):
                    return await func(*args, **kwargs)
            return tmp()
    return wrapper
