import os

from random import randrange, shuffle
from typing import List
from pydantic import Field
from asgi_yauth.settings import AsgiYauthConfig
import logging
import json
logger = logging.getLogger(__name__)

env_type = Field('development', env='ENV_TYPE')
REGRU_CONFIG = {}

try:
    REGRU_CONFIG = json.loads(os.getenv('REGRU_CONFIG'))
except Exception as error:
    if env_type == 'production':
        # do not start production application without regru config
        raise Exception('error getting REGRU_CONFIG ' + str(error))
    else:
        REGRU_CONFIG = {
            'host': 'https://api.reg.ru',
            'username': 'test',
            'password': 'test',
            'contacts': {
                'sms_security_number': '79999999999',
                'p_addr': '119021, Москва, ул. Льва Толстого, 16',
                'org_r': 'ООО Яндекс',
                'phone': ' 79999999900',
                'fax': '+7.9665443322',
                'e_mail': 'e_mail@ya.ru',
                'org': 'yandex',
                'code': '7701107259',
                'kpp': '632946014',
                'country': 'RU',
                'address_r': '101000, Москва, ул.Пупкина, 1, стр. 2'
            },
            'nss': {'ns1': 'dns1.yandex.net'}
        }


class HostManager(object):
    # Maps datacentres to their `nearest neighbours` in a preferred order.
    # Default ordering is by geodesic distance.
    # See github.yandex-team.ru/gist/mkznts/8f25c19950c0e2a959694e4e66eedec7
    PREFERRED_DCS = {
        'iva': ('iva', 'myt', 'vla', 'sas', 'man'),
        'myt': ('myt', 'iva', 'vla', 'sas', 'man'),
        'sas': ('sas', 'vla', 'iva', 'myt', 'man'),
        'vla': ('vla', 'iva', 'myt', 'sas', 'man'),
        'man': ('man', 'myt', 'iva', 'vla', 'sas'),
    }

    def __init__(self, hosts, current_dc=None, preferred_dcs=None):
        self._hosts = hosts
        self._current_dc = current_dc
        dcs = preferred_dcs or self.PREFERRED_DCS
        try:
            preferred = dcs[self.current_dc]
        except KeyError:
            # Use random order when the current DC is unknown or undefined
            preferred = list(dcs.keys())
            shuffle(preferred)
        # Map DCs to ranks. The less the more preferred.
        self.ranks = {dc: rank for rank, dc in enumerate(preferred)}

    @property
    def current_dc(self):
        if self._current_dc is not None:
            return self._current_dc
        dc = (os.getenv('QLOUD_DATACENTER')
              or os.getenv('DEPLOY_NODE_DC')) or ''
        return dc.lower()

    @property
    def host_string(self):
        return ','.join(self.sorted_hosts)

    @property
    def sorted_hosts(self):
        s = sorted(self._hosts, key=lambda x: self.get_rank(x[1]))
        return [host for host, dc in s]

    def get_rank(self, dc):
        return self.ranks.get(dc, randrange(0, len(self.ranks) + 1))

    @classmethod
    def create_from_yc(cls, yc_hosts, current_dc=None):
        """
        Create using list of hosts as returned by `yc mdb cluster ListHosts`:
        {
            "name": <hostname>,
            "options": {"geo": "sas|vla|man|..."}
        }
        """
        host_geo = [
            (host['name'], host.get('options', {}).get('geo'))
            for host in yc_hosts
        ]
        return cls(hosts=host_geo, current_dc=current_dc)


class Settings(AsgiYauthConfig):
    env_type: str = env_type
    debug: bool = False
    docs_prefix: str = '/docs'
    ping_path: str = '/api/common/ping'

    registrar_get_token_secret: bytes
    registrar_store_token_secret: bytes
    registrar_store_password_secret: bytes

    db_drivername: str = 'postgresql'
    db_use_connection_for_request: bool = True
    db_retry_limit: int = 5
    db_retry_times: int = 20
    db_retry_interval: int = 1
    db_pool_min_size: int = 5
    db_pool_max_size: int = 10
    db_username: str
    db_password: str
    db_name: str
    db_port: int = 6432
    db_master_host: str = ''
    db_raw_host: List[dict]
    db_ssl: bool = True
    db_echo: bool = False
    db_sslmode: str = 'verify-full'

    deploy_pod_id: str = None

    webmaster_host: str = 'https://webmaster3-internal.test.in.yandex.net'
    webmaster_tvm2_client: str = '2000286'

    gendarme_host: str = 'https://test.gendarme.mail.yandex.net'
    gendarme_tvm2_client: str = '2001650'

    passport_host = 'http://passport-test-internal.yandex.ru'

    blackbox_url = 'http://pass-test.yandex.ru/blackbox'
    blackbox_tvm2_client: str = '224'

    fouras_host: str = 'https://test.fouras.mail.yandex.net'
    fouras_tvm2_client: str = '2010950'

    connect_host: str = 'https://api-internal-test.directory.ws.yandex.net'
    connect_tvm2_client: str = '2000204'

    gozora_tvm2_client: str = '2023123'

    yandex_team_org_domains: set = {'yandex-team.ru', 'yandex.ru', }
    tech_domain_part = '.yaconnect.com'

    passport_tvm2_client: str = '2000079'

    mailsettings_host: str = 'https://settings-test.mail.yandex.net'
    mailsettings_tvm2_client: str = '2002474'

    sender_host: str = 'https://test.sender.yandex-team.ru'
    sender_tvm2_client: str = '2000074'

    domains_blacklist = [
        'yandex.info',
    ]

    events_saving_enabled = True

    redis_sentinel: bool = True
    redis_sentinel_master: str = 'domenator-test'
    redis_host: str = 'localhost'
    redis_port: str = '6379'
    redis_password: str = 'pass'
    redis_queue_name: str = 'domenator-queue'

    logbroker_write_endpoint: str = ''
    logbroker_port: str = ''
    logbroker_topic: str = ''
    logbroker_tvm2_client: str = '2001059'

    mail_domains_shared_admin_uid = '4004462735'
    mail_allowed_uids = '1422773883,1412277073'
    regru_config: dict = REGRU_CONFIG

    users_with_emulation: List[str] = []
    users_allowed_to_set_register_ts: List[str] = []

    authoritative_dns: List[str] = [
        # '2001:678:17:0:193:232:128:6',  # a.dns.ripn.net
        '2001:678:16:0:194:85:252:62',  # b.dns.ripn.net
        '2001:678:18:0:194:190:124:17',  # d.dns.ripn.net
        '2001:678:15:0:193:232:142:17',  # e.dns.ripn.net
        '2001:678:14:0:193:232:156:17',  # f.dns.ripn.net
    ]
    dns_lifetime: int = 2

    domain_cache = set()
    domain_cache_mtime: int = 0

    connect_check_domain_percent: int = int(os.getenv('CONNECT_CHECK_DOMAIN_PERCENTAGE', 100))
    connect_check_domain_in_bb_percent: int = int(os.getenv('CONNECT_CHECK_DOMAIN_IN_BB_PERCENTAGE', 0))

    @property
    def db_host(self):
        if isinstance(self.db_raw_host, list):
            # [{'name': 'sas-...', 'geo': {'sas'}},]
            return HostManager.create_from_yc(self.db_raw_host).host_string
        return self.db_raw_host

    class Config:
        env_prefix = ''

    enable_domain_sync: bool = True
    enable_logbroker_task: bool = False
    enable_renew_domains: bool = True
    enable_expired_at: bool = True

    days_for_prolong: int = 40
