# coding: utf-8

from __future__ import unicode_literals

import attr
from cache_utils.decorators import cached
from django.conf import settings
from ids.services.calendar_internal.connector import CalendarInternalConnector
from more_itertools import chunked

from easymeeting.lib.tvm2_client import get_service_ticket


class CalendarConnector(CalendarInternalConnector):
    default_connect_timeout = 8

    # TODO: в ids добавить
    url_patterns = {
        'schedule': 'get-resources-schedule',
        'office': 'get-offices',
        'person': 'get-user-settings',
        'events': 'get-events-brief',
        'persons_events': 'get-availability-intervals',
        'event': 'get-event',
        'all_rooms': 'find-resources',
        'create_event': 'create-event',
    }

    def __init__(self, user_ticket, *args, **kwargs):
        kwargs['user_ticket'] = user_ticket
        kwargs['service_ticket'] = get_service_ticket(settings.CALENDAR_TVM_CLIENT_ID)
        kwargs['user_agent'] = settings.SERVICE
        super(CalendarConnector, self).__init__(*args, **kwargs)


@attr.s(frozen=True)
class Event(object):
    date_from = attr.ib()
    date_to = attr.ib()
    name = attr.ib(default='Без названия')
    description = attr.ib(default='')
    organizer = attr.ib(default='')
    attendees = attr.ib(default=[])

    @classmethod
    def from_data(cls, logins, rooms, **kwargs):
        emails = ['%s@yandex-team.ru' % i for i in logins + rooms]
        return cls(
            organizer=emails[0],
            attendees=emails[1:],
            **kwargs
        )


class CalendarController(object):

    def __init__(self, uid, user_ticket):
        self.uid = uid
        self.calendar_connector = CalendarConnector(user_ticket)

    def get_meetings_schedule(self, date_from, date_to, office_ids):
        return self.calendar_connector.get(
            resource='schedule',
            params={
                'uid': self.uid,
                'officeId': office_ids,
                'from': date_from.isoformat(),
                'to': date_to.isoformat(),
                'bookableOnly': 'true',
                'tz': 'UTC',
            },
        )

    def get_events(self, event_ids):
        event_ids_chunks = chunked(event_ids, settings.EVENT_IDS_CHUNK_SIZE)
        result = []
        for event_ids_chunk in event_ids_chunks:
            result += self.get_events_chunk(event_ids_chunk)
        return result

    def get_events_chunk(self, event_ids):
        if not event_ids:
            return []

        events_data = self.calendar_connector.get(
            resource='events',
            params={
                'uid': self.uid,
                'eventIds': event_ids,
                'forResource': 'true',
            },
        )
        return events_data.get('events', [])

    def get_event(self, event_id):
        event = self.calendar_connector.get(
            resource='event',
            params={
                'uid': self.uid,
                'eventId': event_id,
                'tz': 'UTC',
            },
        )
        if 'id' in event:
            return event

        events = self.get_events([event_id])
        return events[0] if len(events) else {}

    def get_persons_events(self, emails, date_from, date_to, tz=None):
        return self.calendar_connector.get(
            resource='persons_events',
            params={
                'uid': self.uid,
                'emails': emails,
                'from': date_from.isoformat(),
                'to': date_to.isoformat(),
                'idsOnly': 'true',
                'display': 'events',
                'tz': tz,
            },
        )

    def get_all_rooms(self):
        return self._get_all_rooms(self.uid)

    def create_event(self, event):
        return self.calendar_connector.post(
            resource='create_event',
            params={
                'uid': self.uid,
                'tz': 'UTC',
            },
            json={
                'type': 'user',
                'availability': 'busy',
                'startTs': event.date_from.isoformat(),
                'endTs': event.date_to.isoformat(),
                'name': event.name,
                'description': event.description,
                'organizer': event.organizer,
                'attendees': event.attendees,
            },
        )

    @cached(
        timeout=60 * 60 * 24,
        key='get_all_rooms',
    )
    def _get_all_rooms(self, uid):
        """
        :param uid: Передаем явно, чтобы по нему строился хэш для кеширования ответа
        """
        assert uid == self.uid

        rooms = self.calendar_connector.get(
            resource='all_rooms',
            params={
                'uid': uid,
            },
        )
        return [
            room['email']
            for room in rooms.get('resources', [])
            if 'email' in room
        ]
