# coding: utf-8

from __future__ import unicode_literals, division

import attr
from cached_property import cached_property
from django.conf import settings

from easymeeting.core import const
from easymeeting.core import schedules
from easymeeting.core import persons_availability
from easymeeting.core import offices
from easymeeting.lib import datetimes

COMPLEMENT_COMBINATIONS_MAX_COUNT = 5


@attr.s(frozen=True)
class Combination(object):
    slots = attr.ib(
        default=attr.Factory(tuple),
        converter=tuple,
    )

    @cached_property
    def interval(self):
        """Слоты всегда идут непрерывно"""
        return (
            min(slot.date_from for slot in self.slots),
            max(slot.date_to for slot in self.slots)
        )

    @cached_property
    def date_from(self):
        return self.interval[0]

    @cached_property
    def date_to(self):
        return self.interval[1]

    @cached_property
    def duration(self):
        return (self.date_to - self.date_from).total_seconds() / 60

    @cached_property
    def factors(self):
        hops_duration = (len(self.slots) - 1) * settings.HOP_DURATION
        result = {
            const.FACTOR_PROPERTIES.HOPS: int(const.FACTOR_VALUES.MAX_SCALE_VALUE * hops_duration / self.duration)
        }

        for slot in self.slots:
            part = datetimes.get_part_of_interval(slot.interval, self.interval)
            for factor_name, factor_value in slot.factors.items():
                if factor_name not in result:
                    result[factor_name] = 0
                result[factor_name] += part * factor_value

        for key, value in result.items():
            result[key] = int(result[key])
        return result


@attr.s(frozen=True)
class RoomSlotBase(object):
    room = attr.ib()
    interval = attr.ib()

    @cached_property
    def date_from(self):
        return self.interval[0]

    @cached_property
    def date_to(self):
        return self.interval[1]

    @cached_property
    def factors(self):
        PROPS = const.FACTOR_PROPERTIES
        VALUES = const.FACTOR_VALUES
        persons_availability_factor = (
            self.persons_availability.availability
            if self.persons_availability is not None
            else 0
        )
        return {
            PROPS.BOOKED: VALUES.MAX_SCALE_VALUE * int(self.booked),
            PROPS.PERSONS_AVAILABILITY: int(VALUES.MAX_SCALE_VALUE * persons_availability_factor),
        }

    @cached_property
    def info(self):
        if self.persons_availability is None:
            return {}
        PROPS = const.INFO_PROPERTIES
        return {
            PROPS.UNAVAILABLE_PERSONS_COUNT: self.persons_availability.unavailable_count,
            PROPS.TOTAL_PERSONS_COUNT: self.persons_availability.total_count,
        }


@attr.s(frozen=True)
class VacantSlot(RoomSlotBase):
    booked = False
    event = None
    persons_availability = None
    event_id = None


@attr.s(frozen=True)
class BookedSlot(RoomSlotBase):
    booked = True
    event = attr.ib()

    @property
    def persons_availability(self):
        return self.event.persons_availability

    @property
    def event_id(self):
        return self.event.event_id


def prepare_combinations(interval, raw_schedule):
    result = {}
    for office_data in raw_schedule['offices']:
        office_id = office_data['id']
        office_schedule = schedules.OfficeSchedule(
            room_schedules=[
                schedules.RoomSchedule.from_raw_resource(raw_resource)
                for raw_resource in office_data['resources']
            ]
        )
        result[office_id] = prepare_office_combinations(interval, office_schedule)
    return result


def prepare_office_combinations(interval, office_schedule):
    combinations = []
    rooms_data = office_schedule.get_rooms_with_intervals(
        interval=interval,
        order_by='-vacant_coverage',
    )
    for room_data in rooms_data:
        if not room_data.booked:
            combinations.append(_build_single_vacant_combination(room_data))
            continue
        else:
            combinations.append(_get_single_room_multislot_combination(room_data))
            if room_data.vacant:
                combinations.extend(
                    _get_multiroom_combinations(room_data, office_schedule)
                )
    combinations = list(set(combinations))
    return combinations


def _build_single_vacant_combination(room_data):
    """
    Комбинация, где переговорка целиком свободна на нужное время
    """
    only_vacant_slice = room_data.vacant[0]
    return Combination(
        slots=[
            VacantSlot(
                room=room_data.room,
                interval=only_vacant_slice.interval,
            ),
        ]
    )


def _get_single_room_multislot_combination(room_data):
    """
    Комбинация, состоящая из свободных и занятых слотов одной и той же
    переговорки.
    """
    slots = []
    for vacant in room_data.vacant:
        slots.append(VacantSlot(
            room=room_data.room,
            interval=vacant.interval,
        ))
    for booked in room_data.booked:
        slots.append(BookedSlot(
            room=room_data.room,
            interval=booked.interval,
            event=booked.event,
        ))
    slots.sort(key=lambda slot: slot.date_from)
    return Combination(slots=slots)


def _get_multiroom_combinations(room_data, office_schedule):
    """
    Несколько комбинаций из нескольких свободных переговорок,
    вместе они покрывают нужный интервал целиком.
    """
    # пока подбираем варианты не более, чем из двух переговорок,
    # более сложные случаи отложим на потом — там получается много
    # комбинаций и применимость ограничена
    combinations = []
    if len(room_data.booked) > 1:
        return []

    initial_room = room_data.room
    initial_room_only_booked = room_data.booked[0]
    initial_room_vacant = room_data.vacant
    complement_rooms = office_schedule.get_rooms_with_intervals(
        interval=initial_room_only_booked.interval,
        order_by='-vacant_coverage',
    )[:COMPLEMENT_COMBINATIONS_MAX_COUNT]

    initial_room_vacant_slots = [
        VacantSlot(
            room=initial_room,
            interval=vacant.interval,
        ) for vacant in initial_room_vacant
    ]
    for complement_room_data in complement_rooms:
        if not complement_room_data.booked:
            only_complement_vacant = complement_room_data.vacant[0]
            slots = sorted(
                initial_room_vacant_slots + [
                    VacantSlot(
                        room=complement_room_data.room,
                        interval=only_complement_vacant.interval,
                    ),
                ],
                key=lambda slot: slot.date_from,
            )
            combinations.append(Combination(slots=slots))
    return combinations


def get_event_ids_by_offices(raw_schedule):
    event_ids_by_offices = {}
    for office in raw_schedule.get('offices', []):
        event_ids = []
        for resource in office.get('resources', []):
            for event in resource.get('events', []):
                event_id = event.get('eventId')
                if event_id:
                    event_ids.append(event_id)
        office_id = office.get('id')
        if office_id:
            event_ids_by_offices[office_id] = event_ids
    return event_ids_by_offices


def get_persons_from_event(event):
    attendees = [attender for attender in event.get('attendees', [])]
    attendees.append(event.get('organizer'))

    # Исключаем рассылки (участников без логина)
    return [i for i in attendees if i and 'login' in i]


def get_persons_by_events(events):
    persons_by_events = {}
    for event in events:
        event_id = event.get('id')
        persons_by_events[event_id] = get_persons_from_event(event)
    return persons_by_events


def get_persons_logins(persons_by_events):
    logins = set()
    for persons in persons_by_events.values():
        event_logins = [
            person.get('login')
            for person in persons
            if 'login' in person
        ]
        logins.update(event_logins)
    return list(logins)


def get_person_office_id(staff_person, calendar_person):
    home_office_id = staff_person.get('location', {}).get('office', {}).get('id')
    last_office_id = calendar_person.get('officeId')
    return home_office_id if home_office_id else last_office_id


def is_person_unavailable(staff_person, person_gaps):
    is_dismissed = staff_person.get('official', {}).get('is_dismissed', False)
    has_gaps = len(person_gaps) > 0
    return has_gaps or is_dismissed


def get_persons_availability(calendar_persons_by_events,
                             persons_gap_by_logins,
                             staff_persons_by_logins,
                             event_ids_by_offices):
    availability_by_offices = {}
    for office_id in event_ids_by_offices:
        availability_by_events = {}
        for event_id in event_ids_by_offices[office_id]:
            office_person_logins = []
            for calendar_person in calendar_persons_by_events[event_id]:
                login = calendar_person.get('login')
                staff_person = staff_persons_by_logins.get(login, {})
                person_office_id = get_person_office_id(
                    staff_person=staff_person,
                    calendar_person=calendar_person,
                )
                in_office = offices.is_nearly(office_id, person_office_id)
                is_confirmed = calendar_person.get('decision') != 'no'
                if in_office and is_confirmed:
                    office_person_logins.append(login)
            unavailable = [
                login
                for login in office_person_logins
                if is_person_unavailable(
                    staff_person=staff_persons_by_logins.get(login, {}),
                    person_gaps=persons_gap_by_logins.get(login, {}),
                )
            ]
            available = [
                login
                for login in office_person_logins
                if login not in unavailable
            ]
            availability_by_events[event_id] = persons_availability.PersonsAvailability(
                available=available,
                unavailable=unavailable,
            )
        availability_by_offices[office_id] = availability_by_events
    return availability_by_offices


def merge_persons_availability(raw_schedule, persons_availability):
    for office in raw_schedule.get('offices', []):
        office_id = office.get('id')
        for resource in office.get('resources', []):
            for event in resource.get('events', []):
                event_id = event.get('eventId')
                if event_id and office_id:
                    event['persons_availability'] = persons_availability[office_id][event_id]


def filter_bookable_rooms(raw_schedule, bookable_rooms, interval):
    for office in raw_schedule.get('offices', []):
        resources = []
        for resource in office.get('resources', []):
            if resource.get('info', {}).get('email') not in bookable_rooms:
                continue
            if _get_is_restricted(resource, interval):
                continue
            resources.append(resource)
        office['resources'] = resources


def _get_is_restricted(resource, event_interval):
    # restrictions используются для вывода переговорки из доступности на время ремонта или праздника
    # Это довольно длительные интервалы времени, поэтому исключаем ее сразу из всех стадий
    for restriction in resource.get('restrictions', []):
        restriction_start = (
            datetimes.parse_calendar_datetime_str(restriction['start'])
            if 'start' in restriction
            else event_interval[0]
        )
        restriction_end = (
            datetimes.parse_calendar_datetime_str(restriction['end'])
            if 'end' in restriction
            else event_interval[1]
        )
        if datetimes.get_intersection(event_interval, (restriction_start, restriction_end)):
            return True
    return False
