# coding: utf-8

from __future__ import unicode_literals

import attr
from cache_utils.decorators import cached
from django.conf import settings
from django.db import transaction

from easymeeting.core import models
from easymeeting.core.calendar import CalendarConnector


class Office(object):
    def __init__(self, **kwargs):
        self.id = kwargs['id']
        self.name = kwargs['name']
        self.city_name = kwargs['cityName']
        self.tz_offset = kwargs['tzOffset']


Office = attr.s(
    these={
        'id': attr.ib(),
        'name': attr.ib(),
        'city_name': attr.ib(),
        'tz_offset': attr.ib(),
    },
    init=False,
)(Office)


class OfficeController(object):

    def __init__(self, uid, user_ticket):
        self.uid = uid
        self.calendar_connector = CalendarConnector(user_ticket)

    def get_all_offices(self, language='ru'):
        return self._get_all_offices(self.uid, language)

    def get_offices(self, office_ids, language='ru'):
        offices = self.get_all_offices(language)
        filtered_offices = self._filter_offices(offices, office_ids)
        return [Office(**data) for data in filtered_offices]

    @transaction.atomic
    def collect_offices_from_staff_and_calendar(self):
        models.Office.objects.all().delete()
        calendar_offices = self.get_all_offices()
        office_models = []
        for office_data in calendar_offices:
            office_models.append(models.Office(
                calendar_id=office_data['id'],
                name_ru=office_data['name'],
                city_name_ru=office_data['cityName'],
            ))
        models.Office.objects.bulk_create(office_models)

    @cached(
        timeout=60 * 60 * 24,
        key='get_all_offices',
    )
    def _get_all_offices(self, uid, language):
        """
        :param uid: Передаем явно, чтобы по нему строился хэш для кеширования ответа
        :param language: 'ru' или 'en'
        """
        assert uid == self.uid

        response = self.calendar_connector.get(
            resource='office',
            params={
                'uid': uid,
                'includeOthers': 'false',
                'lang': language,
                'tz': 'UTC',
            },
        )
        return response['offices']

    def _filter_offices(self, offices, ids):
        if not ids:
            return offices

        return [
            office
            for office in offices
            if office['id'] in ids
        ]


def is_nearly(office_id_a, office_id_b):
    return (
        (office_id_a in settings.RED_ROSE_OFFICES and office_id_b in settings.RED_ROSE_OFFICES)
        or office_id_a == office_id_b
    )
