# coding: utf-8

from __future__ import unicode_literals

import attr
from django.conf import settings
from ids.registry import registry

from easymeeting.core.helpers import safe_getitem
from easymeeting.lib.tvm2_client import get_service_ticket


class Person(object):

    def __init__(self, **kwargs):
        self.uid = kwargs['uid']
        self.login = kwargs['login']
        self.name = kwargs['name']
        self.current_office_id = kwargs['currentOfficeId']


Person = attr.s(
    these={
        'uid': attr.ib(),
        'login': attr.ib(),
        'name': attr.ib(),
        'current_office_id': attr.ib(),
    },
    init=False,
)(Person)


def get_persons(uids, language='ru'):
    """
    :return: List(Person)
    """
    persons = _get_staff_persons_raw(
        uid=','.join(str(uid) for uid in uids),
    )
    alternative_language = 'en' if language == 'ru' else 'ru'
    person_by_uid_dict = {
        int(p['uid']): Person(
            uid=int(p['uid']),
            login=p['login'],
            name=_get_name(p, language) or _get_name(p, alternative_language),
            currentOfficeId=_get_office_id(p),
        ) for p in persons
    }
    return [
        person_by_uid_dict[uid]
        for uid in uids
    ]


def get_person_by_login_dict(logins):
    """
    :return: {login: staff_person_dict_raw}
    """
    if not logins:
        return {}
    persons = _get_staff_persons_raw(
        login=','.join(logins),
        _fields=','.join([
            'login',
            'location.office.id',
            'official.is_dismissed',
        ]),
    )
    persons_by_logins = {}
    for person in persons:
        login = person.get('login')
        persons_by_logins[login] = person
    # TODO: отдавать class Person аналогично get_persons
    return persons_by_logins


def _get_staff_persons_raw(**kwargs):
    """
    :param kwargs: поля, по которым фильтровать коллекцию Person
    :return: staff_person_dict_raw
    """
    repo = registry.get_repository(
        service='staff',
        resource_type='person',
        user_agent=settings.SERVICE,
        service_ticket=get_service_ticket(settings.STAFF_API_TVM_CLIENT_ID),
        timeout=8,
    )
    kwargs.setdefault('_fields', ','.join([
        'uid',
        'login',
        'name',
        'location.office.id',
        'official.is_dismissed',
    ]))
    kwargs.setdefault('_sort', 'id')
    kwargs.setdefault('_limit', 1000)
    return repo.get(lookup=kwargs)


def _get_name(staff_person, lang):
    return ' '.join((
        safe_getitem(staff_person, ['name', 'first', lang], ''),
        safe_getitem(staff_person, ['name', 'last', lang], ''),
    ))


def _get_office_id(staff_person):
    office_id = safe_getitem(
        staff_person,
        ['location', 'office', 'id'],
        settings.HOMEWORKER_OFFICE_ID,
    )
    # Хотфикс для надомников, пока фронтенд не научится работать с ними (доделаем в EASYMEET-160)
    return office_id if office_id != settings.HOMEWORKER_OFFICE_ID else settings.MOROZOV_OFFICE_ID
