# coding: utf-8

from __future__ import unicode_literals

import attr

from easymeeting.lib import datetimes
from easymeeting.core import rooms
from easymeeting.core import resource_event


@attr.s
class RoomSlice(object):
    interval = attr.ib()
    event = attr.ib(default=None)


@attr.s
class Order(object):
    field = attr.ib()
    reverse = attr.ib(default=False)

    @classmethod
    def from_str(cls, order_string=None):
        if order_string is None:
            return cls(field=None)
        if order_string.startswith('-'):
            reverse = True
            field = order_string[len('-'):]
        elif order_string.startswith('+'):
            reverse = False
            field = order_string[len('-'):]
        else:
            reverse = False
            field = order_string
        return cls(
            reverse=reverse,
            field=field,
        )

    def __nonzero__(self):
        return self.field is not None


@attr.s
class RoomSchedule(object):
    room = attr.ib()
    events = attr.ib()

    @classmethod
    def from_raw_resource(cls, raw_resource):
        events = [
            resource_event.ResourceEvent.from_raw_event(**raw_event)
            for raw_event in raw_resource['events']
        ]
        return cls(
            room=rooms.Room.from_raw_resource(raw_resource),
            events=events
        )

    def is_available_at(self, interval):
        for event in self.events:
            if datetimes.get_intersection(event.interval, interval):
                return False
        return True

    def get_vacant_intersections(self, interval):
        slices = self.get_booked_intersections(interval)
        vacant_intervals = [room_schedule.interval for room_schedule in slices]
        difference = datetimes.substract_intervals(
            minuend=interval,
            subtrahends_list=vacant_intervals,
        )
        intervals = [] if difference == [datetimes.EMPTY_INTERVAL] else difference
        return [RoomSlice(interval=interval) for interval in intervals]

    def get_booked_intersections(self, interval):
        slices = []
        for event in self.events:
            intersection = datetimes.get_intersection(interval, event.interval)
            if intersection:
                item = RoomSlice(
                    interval=intersection,
                    event=event,
                )
                slices.append(item)
        return slices


@attr.s
class RoomData(object):
    room = attr.ib()
    vacant = attr.ib(attr.Factory(list))
    booked = attr.ib(attr.Factory(list))

    def get_vacant_coverage(self, interval):
        coverage = 0
        for vacant in self.vacant:
            coverage += datetimes.get_part_of_interval(vacant.interval, interval)
        return coverage


@attr.s
class OfficeSchedule(object):

    room_schedules = attr.ib(default=attr.Factory(list))

    def get_rooms_with_intervals(self, interval, order_by=None):
        result = []
        for room_schedule in self.room_schedules:
            result.append(
                RoomData(
                    room=room_schedule.room,
                    vacant=room_schedule.get_vacant_intersections(interval),
                    booked=room_schedule.get_booked_intersections(interval),
                )
            )

        order_by = Order.from_str(order_by)
        if order_by.field == 'vacant_coverage':
            result.sort(
                key=lambda rd: rd.get_vacant_coverage(interval),
                reverse=order_by.reverse,
            )
        return result
