# coding: utf-8
from __future__ import unicode_literals

import json

from django import forms
from django.conf import settings

from easymeeting.core.calendar import CalendarController
from easymeeting.core.offices import OfficeController
from easymeeting.lib import views
from easymeeting.lib import datetimes
from easymeeting.lib.tvm2_client import get_robot_almighty_user_ticket
from easymeeting.core import combinations
from easymeeting.core import gaps
from easymeeting.core import person
from easymeeting.core import models


class ParticipantForm(forms.Form):
    login = forms.CharField(required=False)
    officeId = forms.IntegerField(required=False)

    def clean(self):
        cleaned_data = super(ParticipantForm, self).clean()
        if not cleaned_data['login'] and not cleaned_data['officeId']:
            raise forms.ValidationError('login or officeId required')
        return cleaned_data


class CombinationForm(forms.Form):
    dateFrom = forms.DateTimeField()
    dateTo = forms.DateTimeField()

    def clean(self):
        cleaned_data = super(CombinationForm, self).clean()
        participants = self.data.get('participants')
        if not participants:
            self._errors = {'participants': 'participants list required'}
            return
        participants_forms = [
            ParticipantForm(data=participant_data)
            for participant_data in participants
        ]
        participants_errors = [
            form.errors for form in participants_forms
            if not form.is_valid()
        ]
        if participants_errors:
            self._errors = {'participants': participants_errors}

        cleaned_data['participants'] = [
            form.cleaned_data for form in participants_forms]
        return cleaned_data


class CombinationsView(views.View):
    form_cls_post = CombinationForm

    def process_post(self, data):
        save_visit(
            uid=self.request.yauser.uid,
            data=data,
        )
        office_ids = [
            participant_data['officeId']
            for participant_data in data['participants']
            if participant_data['officeId']
        ]
        date_from = data['dateFrom']
        date_to = data['dateTo']
        raw_schedule = self.calendar_controller.get_meetings_schedule(
            date_from=date_from,
            date_to=date_to,
            office_ids=office_ids,
        )
        filter_bookable_rooms(
            calendar_controller=self.calendar_controller,
            raw_schedule=raw_schedule,
            interval=(date_from, date_to),
        )
        add_persons_availability(
            raw_schedule=raw_schedule,
            date_from=date_from,
            date_to=date_to,
        )
        combinations_data = combinations.prepare_combinations(
            interval=(date_from, date_to),
            raw_schedule=raw_schedule,
        )
        offices_data = prepare_offices(
            office_controller=OfficeController(self.request.yauser.uid, self.user_ticket),
            raw_schedule=raw_schedule,
            language=self.request.yauser.fields.get('language', 'ru'),
        )
        return {
            'offices': serialize_offices_with_combinations(
                data={
                    'combinations': combinations_data,
                    'offices': offices_data
                },
            )
        }


def save_visit(uid, data):
    models.Visit.objects.create(
        uid=uid,
        params=json.dumps({
            'dateFrom': datetimes.datetime_to_str(data['dateFrom']),
            'dateTo': datetimes.datetime_to_str(data['dateTo']),
            'participants': data['participants'],
        }),
    )


def filter_bookable_rooms(calendar_controller, raw_schedule, interval):
    bookable_rooms = calendar_controller.get_all_rooms()
    combinations.filter_bookable_rooms(raw_schedule, bookable_rooms, interval)


def add_persons_availability(raw_schedule, date_from, date_to):
    event_ids_by_offices = combinations.get_event_ids_by_offices(raw_schedule)
    event_ids = [
        event_id
        for office_id in event_ids_by_offices
        for event_id in event_ids_by_offices[office_id]
    ]

    # Note: ходим за событиями календаря от лица
    # нашего всемогущего робота @robot-easymeeting-2,
    # у которого есть доступ ко всем встречам
    robot_user_ticket = get_robot_almighty_user_ticket()
    controller = CalendarController(settings.ROBOT_ALMIGHTY_UID, robot_user_ticket)
    events = controller.get_events(event_ids)

    calendar_persons_by_events = combinations.get_persons_by_events(events)
    logins = combinations.get_persons_logins(calendar_persons_by_events)
    persons_gap_by_logins = gaps.get_persons_gap_by_logins(
        logins=logins,
        date_from=date_from,
        date_to=date_to,
    )
    staff_persons_by_logins = person.get_person_by_login_dict(logins)
    persons_availability = combinations.get_persons_availability(
        calendar_persons_by_events=calendar_persons_by_events,
        persons_gap_by_logins=persons_gap_by_logins,
        staff_persons_by_logins=staff_persons_by_logins,
        event_ids_by_offices=event_ids_by_offices,
    )
    combinations.merge_persons_availability(
        raw_schedule=raw_schedule,
        persons_availability=persons_availability,
    )


def prepare_offices(office_controller, raw_schedule, language='ru'):
    office_ids = list({
        o['id'] for o in raw_schedule['offices']
    })
    fetched = office_controller.get_offices(office_ids=office_ids, language=language)
    return {o.id: o for o in fetched}


def serialize_offices_with_combinations(data):
    result = []
    for office_id, office_obj in data['offices'].items():
        office_with_combinations = {
            'id': office_obj.id,
            'name': office_obj.name,
            'city': {
                'name': office_obj.city_name,
            },
            'combinations': map(
                serialize_combination,
                data['combinations'].get(office_id, [])
            )
        }
        result.append(office_with_combinations)
    return result


def serialize_combination(combination):
    return {
        'factors': combination.factors,
        'slots': map(serialize_slot, combination.slots)
    }


def serialize_slot(slot):
    return {
        'factors': slot.factors,
        'info': slot.info,
        'dateFrom': slot.date_from,
        'dateTo': slot.date_to,
        'name': {
            'display': slot.room.display_name,
            'exchange': slot.room.exchange_name,
        },
        'eventId': slot.event_id,
    }
