# coding: utf-8

from __future__ import unicode_literals

from django import forms
from rest_framework import serializers

from easymeeting.core import calendar
from easymeeting.core import combinations
from easymeeting.core import gaps
from easymeeting.core import person
from easymeeting.lib import views
from easymeeting.lib import datetimes


class GetEventsForm(forms.Form):
    eventIds = forms.CharField()

    def clean_eventIds(self):
        event_ids = self.cleaned_data['eventIds']
        return map(int, event_ids.split(','))


class SlotSerializer(serializers.Serializer):
    dateFrom = serializers.DateTimeField()
    dateTo = serializers.DateTimeField()
    exchange = serializers.CharField(required=False)


class EventsSerializer(serializers.Serializer):
    name = serializers.CharField(required=False, allow_blank=True)
    description = serializers.CharField(required=False, allow_blank=True)
    logins = serializers.ListField(child=serializers.CharField(), min_length=1)
    slots = serializers.ListField(child=SlotSerializer(), min_length=1)


def get_event_and_persons_gap(calendar_controller, event_id):
    event = calendar_controller.get_event(event_id)
    logins = [
        calendar_person.get('login')
        for calendar_person in combinations.get_persons_from_event(event=event)
    ]
    date_from = datetimes.parse_calendar_datetime_str(event.get('startTs'))
    date_to = datetimes.parse_calendar_datetime_str(event.get('endTs'))
    persons_gap_by_logins = gaps.get_persons_gap_by_logins(
        logins=logins,
        date_from=date_from,
        date_to=date_to,
    ) if date_to and date_from else {}
    staff_persons_by_logins = person.get_person_by_login_dict(logins)
    return serialize_event(
        event=event,
        persons_gap_by_logins=persons_gap_by_logins,
        staff_persons_by_logins=staff_persons_by_logins,
    )


def serialize_event(event, persons_gap_by_logins, staff_persons_by_logins):
    organizer = serialize_person(
        calendar_person=event.get('organizer'),
        persons_gap_by_logins=persons_gap_by_logins,
        staff_persons_by_logins=staff_persons_by_logins,
    )
    attendees = [
        serialize_person(
            calendar_person=attendee,
            persons_gap_by_logins=persons_gap_by_logins,
            staff_persons_by_logins=staff_persons_by_logins,
        )
        for attendee in event.get('attendees', [])
        # Исключаем рассылки (участников без логина)
        if attendee.get('login')
    ]
    return {
        'id': event.get('id'),
        'name': event.get('name'),
        'description': event.get('descriptionHtml'),
        'dateFrom': datetimes.parse_calendar_datetime_str(event.get('startTs')),
        'dateTo': datetimes.parse_calendar_datetime_str(event.get('endTs')),
        'organizer': organizer,
        'attendees': attendees,
    }


def serialize_person(calendar_person, persons_gap_by_logins, staff_persons_by_logins):
    if not calendar_person:
        return {}
    login = calendar_person.get('login')
    gaps = persons_gap_by_logins.get(login, [])
    officeId = combinations.get_person_office_id(
        staff_person=staff_persons_by_logins.get(login, {}),
        calendar_person=calendar_person
    )
    return {
        'login': login,
        'name': calendar_person.get('name'),
        'gaps': gaps,
        'officeId': officeId,
        'decision': calendar_person.get('decision'),
    }


class EventsView(views.View):
    form_cls_get = GetEventsForm
    form_cls_post = EventsSerializer

    def process_get(self, data):
        events = [
            get_event_and_persons_gap(self.calendar_controller, event_id)
            for event_id in data['eventIds']
        ]
        return {'events': events}

    def process_post(self, data):
        name = data.get('name', '')
        intervals = get_intervals(data['slots'])
        events = []
        count = len(intervals)
        for part, interval in enumerate(intervals, 1):
            rooms = get_rooms_by_interval(
                interval=interval,
                slots=data['slots'],
            )
            date_from, date_to = interval
            event = calendar.Event.from_data(
                date_from=date_from,
                date_to=date_to,
                name=get_event_name(name, part, count),
                description=data.get('description', ''),
                logins=data['logins'],
                rooms=rooms,
            )
            events.append(self.calendar_controller.create_event(event))
        return events


def get_intervals(slots):
    dates = set()
    for slot in slots:
        dates.add(slot.get('dateFrom'))
        dates.add(slot.get('dateTo'))
    ordered_dates = sorted(dates)
    intervals = []
    start_date = ordered_dates.pop(0)
    for end_date in ordered_dates:
        intervals.append((start_date, end_date))
        start_date = end_date
    return intervals


def get_rooms_by_interval(interval, slots):
    return [
        slot['exchange']
        for slot in slots
        if (
            slot.get('exchange')
            and datetimes.is_include_interval((slot['dateFrom'], slot['dateTo']), interval)
        )
    ]


def get_event_name(name, part, count):
    name = 'Без названия' if not name else name
    suffix = ' [часть {} из {}]'.format(part, count) if count > 1 else ''
    return name + suffix
