# coding: utf-8

from __future__ import unicode_literals

import itertools

import pytz

from datetime import timedelta

from django import forms
from django.conf import settings
from django.core.exceptions import ValidationError
from rest_framework import serializers

from easymeeting.core import person, person_event, gaps
from easymeeting.lib import views, datetimes


class PersonForm(forms.Form):
    uids = forms.CharField()
    dateFrom = forms.DateTimeField()
    dateTo = forms.DateTimeField()

    def clean_uids(self):
        uids = self.cleaned_data['uids']
        return map(int, uids.split(','))


class ParticipantSerializer(serializers.Serializer):

    login = serializers.CharField()


class PersonFreeIntervalSerializer(serializers.Serializer):

    duration = serializers.IntegerField()
    dateFrom = serializers.DateTimeField(required=False)
    dateTo = serializers.DateTimeField(required=False)
    freeTimeFrom = serializers.TimeField(required=False)
    freeTimeTo = serializers.TimeField(required=False)
    participants = ParticipantSerializer(many=True)

    def _validate_time(self, value):
        # От фронта время всегда прилетает в UTC
        return value.replace(tzinfo=pytz.utc)

    validate_freeTimeFrom = _validate_time
    validate_freeTimeTo = _validate_time

    def validate(self, data):
        free_time_from = data.get('freeTimeFrom')
        free_time_to = data.get('freeTimeTo')
        if (free_time_from is None) != (free_time_to is None):
            raise ValidationError('freeTimeFrom and freeTimeTo both required')
        return data


class PersonView(views.View):
    form_cls_get = PersonForm

    def process_get(self, data):
        persons = person.get_persons(
            uids=data['uids'],
            language=self.request.yauser.fields.get('language', 'ru'),
        )
        logins = [item.login for item in persons]
        persons_events = person_event.get_persons_events(
            calendar_controller=self.calendar_controller,
            logins=logins,
            date_from=data['dateFrom'],
            date_to=data['dateTo'],
        )
        persons_gap_by_logins = gaps.get_persons_gap_by_logins(
            logins=logins,
            date_from=data['dateFrom'],
            date_to=data['dateTo'],
        )
        result = [
            {
                'uid': item.uid,
                'login': item.login,
                'name': item.name,
                'currentOfficeId': item.current_office_id,
                'events': [serialize_person_event(e) for e in persons_events[item.login]],
                'gaps': [serialize_gap(gap) for gap in persons_gap_by_logins[item.login]],
            }
            for item in persons
        ]
        return {'persons': result}


class PersonFreeIntervalView(views.View):
    """
    Ручка поиска свободного времени участников.
    Отдает свободное время на ближайшие settings.FREE_INTERVALS_SEARCH_DAYS_LIMIT дней
    нарезанное интервалами с заданной длительностью
    и шагом settings.CALENDAR_INTERVAL_STEP
    """
    form_cls_post = PersonFreeIntervalSerializer

    def process_post(self, data):
        logins = [i['login'] for i in data['participants']]
        date_from = data.get('dateFrom') or datetimes.closest_rounded_datetime()
        date_to = (
            data.get('dateTo')
            or date_from + timedelta(days=settings.FREE_INTERVALS_SEARCH_DAYS_LIMIT)
        )
        free_time_from = data.get('freeTimeFrom')
        free_time_to = data.get('freeTimeTo')

        # Если попадают выходные, убираем их,
        # date_to увеличиваем на кол-во выкинутых выходных.
        # Если date_to задан явно фронтом, тогда его не меняем.
        busy_intervals_by_holidays = datetimes.get_holidays(date_from, date_to)
        if not data.get('dateTo'):
            date_to += timedelta(days=len(busy_intervals_by_holidays))

        persons_events = person_event.get_persons_events(
            calendar_controller=self.calendar_controller,
            logins=logins,
            date_from=date_from,
            date_to=date_to,
            tz='UTC',
        )
        persons_gaps = gaps.get_gaps(
            logins=logins,
            date_from=date_from,
            date_to=date_to,
        )

        busy_intervals_by_events = [
            (
                datetimes.parse_calendar_datetime_str(i.start),
                datetimes.parse_calendar_datetime_str(i.end),
            )
            for i in itertools.chain.from_iterable(persons_events.values())
        ]
        busy_intervals_by_gaps = [(i['date_from'], i['date_to']) for i in persons_gaps]

        busy_intervals_by_free_time = []
        if free_time_from is not None and free_time_to is not None:
            busy_intervals_by_free_time = list(
                datetimes.generate_datetime_intervals_by_time(
                    start=date_from,
                    end=date_to,
                    time_start=free_time_to,
                    time_end=free_time_from,
                )
            )

        busy_intervals = (
            busy_intervals_by_events
            + busy_intervals_by_gaps
            + busy_intervals_by_free_time
            + busy_intervals_by_holidays
        )

        free_intervals_gen = datetimes.generate_free_intervals(
            start=date_from,
            end=date_to,
            duration=data['duration'],
            busy_intervals=busy_intervals,
        )

        return {
            'participants': data['participants'],
            'intervals': map(serialize_interval, free_intervals_gen),
        }


def serialize_person_event(event):
    return {
        'eventId': event.event_id,
        'start': event.start,
        'end': event.end,
        'availability': event.availability,
    }


def serialize_gap(gap):
    return {
        'id': gap['id'],
        'dateFrom': datetimes.datetime_to_str(gap['date_from']),
        'dateTo': datetimes.datetime_to_str(gap['date_to']),
        'workflow': gap['workflow'],
        'fullDay': gap['full_day'],
        'login': gap['person_login']
    }


def serialize_interval(interval):
    return {
        'dateFrom': interval[0],
        'dateTo': interval[1],
    }
