# coding: utf-8

from __future__ import unicode_literals

from logging import getLogger

from django.conf import settings
from django.contrib import auth as django_auth
from django_yauth.user import YandexUser


logger = getLogger(__name__)


class FetchLoginAuthBackend(object):

    def authenticate(self, request):
        login = self.fetch_login(request)
        uid = self.fetch_uid(request)
        if login is None or uid is None:
            return
        return YandexUser(
            uid=uid,
            fields={
                'login': login,
                'language': 'ru',
            }
        )

    def fetch_login(self, request):
        return None

    def fetch_uid(self, request):
        return None


class UserFromYandexLoginHeaderAuthBackend(FetchLoginAuthBackend):

    def fetch_login(self, request):
        return request.META.get('HTTP_DEBUG_LOGIN')

    def fetch_uid(self, request):
        return request.META.get('HTTP_DEBUG_UID')


class UserFromYandexLoginCookieAuthBackend(FetchLoginAuthBackend):
    """
    Способ при разработке задать любого пользователя.
    """
    def fetch_login(self, request):
        return request.COOKIES.get('yandex_login')

    def fetch_uid(self, request):
        # Note: настоящая яндексовая кука не содержит подчеркивания: yandexuid
        return request.COOKIES.get('yandex_uid')


class UserFromSettingsAuthBackend(FetchLoginAuthBackend):

    def fetch_login(self, request):
        return getattr(settings, 'YAUTH_DEV_USER_LOGIN', None)

    def fetch_uid(self, request):
        return getattr(settings, 'YAUTH_DEV_USER_UID', None)


class SetDjangoUserMiddleware(object):
    """
    We need it only for Django admin
    """
    def process_request(self, request):
        if not getattr(request, 'yauser', None):
            logger.warning('No yauser in request. Are you authenticated?')
            return
        user_model_cls = django_auth.get_user_model()
        request.user = user_model_cls(
            login=request.yauser.login,
        )
