# coding: utf-8

from __future__ import unicode_literals

import arrow
import math
from decimal import Decimal
import datetime
import pytz

from django.conf import settings
from django.utils import timezone

EMPTY_INTERVAL = object()
ONE_DAY = datetime.timedelta(days=1)


def shifted(dt, **shift):
    shifted_datetime = arrow.get(dt).replace(**shift)
    # isinstance(date, datetime.date) returns True for datetimes
    if is_date(dt):
        return shifted_datetime.date()
    if is_naive(dt):
        return shifted_datetime.naive
    return shifted_datetime.datetime


def is_date(dt):
    if type(dt) is datetime.date:
        return True
    if isinstance(dt, basestring) and len(dt) == len('2018-01-01'):
        return True
    return False


def is_naive(dt):
    if type(dt) is datetime.datetime and dt.tzinfo is None:
        return True
    if isinstance(dt, basestring) and len(dt) == len('2018-01-01T12:00:00'):
        return True
    return False


def datetime_to_str(dt):
    return dt.strftime(settings.MAIN_DATETIME_FORMAT)


def parse_calendar_datetime_str(dt_str, tz=pytz.utc):
    if dt_str:
        dt = datetime.datetime.strptime(dt_str, settings.CALENDAR_DATETIME_FORMAT)
        return tz.localize(dt)


def parse_gap_datetime_str(dt_str, tz=pytz.utc):
    if dt_str:
        dt = datetime.datetime.strptime(dt_str, settings.GAP_DATETIME_FORMAT)
        return tz.localize(dt)


def utc_datetime(*args, **kwargs):
    kwargs['tzinfo'] = pytz.utc
    return datetime.datetime(*args, **kwargs)


def get_intersection(first, second):
    """
    Получить пересечение двух сравнимых интервалов.
    Если интервалы не пересекаются, возвращает None.
    Границы интевала не включаются в интервал.
    """
    first_start, first_end = first
    second_start, second_end = second
    new_start = max(first_start, second_start)
    new_end = min(first_end, second_end)
    if new_start >= new_end:
        return None
    return new_start, new_end


def substract_interval(minuend, subtrahend):
    """
    Вычитает из первого интервала второй.
    Если второй внутри первого в результате получится два интервала.
    """
    intersection = get_intersection(minuend, subtrahend)
    if intersection is None:
        return [minuend]
    if minuend == subtrahend:
        return [EMPTY_INTERVAL]

    first_start, first_end = minuend
    intersection_start, intersection_end = intersection

    difference = []
    if intersection_start != first_start:
        difference.append((first_start, intersection_start))
    if intersection_end != first_end:
        difference.append((intersection_end, first_end))
    return difference


def substract_intervals(minuend, subtrahends_list):
    difference = [minuend]
    for subtrahend in subtrahends_list:
        new_difference = []
        for interval in difference:
            new_difference.extend([
                intrv for intrv in
                substract_interval(
                    minuend=interval,
                    subtrahend=subtrahend,
                )
                if intrv is not EMPTY_INTERVAL
            ])
        difference = new_difference
    return difference or [EMPTY_INTERVAL]


def get_part_of_interval(first, second):
    """
    Какую долю первый интервал составляет от второго
    """
    first_start, first_end = first
    second_start, second_end = second
    if first_start < second_start or first_end > second_end:
        raise Exception('first interval must be nested for second')

    first_length = Decimal((first_end - first_start).seconds)
    second_length = Decimal((second_end - second_start).seconds)
    return first_length / second_length


def shifted_interval(interval, **shift_kwargs):
    left, right = interval
    return (
        shifted(left, **shift_kwargs),
        shifted(right, **shift_kwargs),
    )


def is_include_interval(inclusive, included):
    inclusive_start, inclusive_end = inclusive
    included_start, included_end = included
    return inclusive_start <= included_start and inclusive_end >= included_end


def generate_free_intervals(start, end, duration, busy_intervals, step=settings.CALENDAR_INTERVAL_STEP):
    """
    Генератор свободных интервалов в промежутке между start и end
    длительностью duration, с шагом step
    """
    difference = substract_intervals((start, end), busy_intervals)
    difference = [] if difference == [EMPTY_INTERVAL] else difference

    for interval in difference:
        _start = ceil_time(interval[0], step * 60)
        _end = _start + datetime.timedelta(minutes=duration)
        while is_include_interval(interval, (_start, _end)):
            yield _start, _end
            _start += datetime.timedelta(minutes=step)
            _end += datetime.timedelta(minutes=step)


def round_time(dt, round_to=60, round_func=None):
    """
    Округляет время в dt до round_to секунд.
    По умолчанию, округление честное математическое, но можно
    передать функцию округления (floor, ceil, round).
    По умолчанию округляет до 1 минуты.
    """
    round_func = round_func or round
    seconds = (dt.replace(tzinfo=None) - dt.min).seconds
    rounded_seconds = int(round_func(seconds / float(round_to)) * round_to)
    return dt - datetime.timedelta(seconds=seconds - rounded_seconds, microseconds=dt.microsecond)


def ceil_time(dt, round_to=60):
    return round_time(dt, round_to, math.ceil)


def floor_time(dt, round_to=60):
    return round_time(dt, round_to, math.floor)


def closest_rounded_datetime(dt=None):
    dt = dt or timezone.now()
    return ceil_time(dt, settings.CALENDAR_INTERVAL_STEP * 60)


def set_time(dt, t):
    assert (dt.tzinfo is None) == (t.tzinfo is None)
    dt_tzinfo = dt.tzinfo

    if dt_tzinfo:
        dt = dt.astimezone(t.tzinfo)

    dt = dt.replace(
        hour=t.hour,
        minute=t.minute,
        second=t.second,
        microsecond=t.microsecond,
    )

    if dt_tzinfo:
        dt = dt.astimezone(dt_tzinfo)

    return dt


def generate_datetime_intervals_by_time(start, end, time_start, time_end, include_partial=True):
    delta = datetime.timedelta(1 if time_start >= time_end else 0)
    _start = start - delta

    while _start <= end:
        x = set_time(_start, time_start)
        y = set_time(_start + delta, time_end)

        intersection = get_intersection((start, end), (x, y))
        if intersection == (x, y) or intersection and include_partial:
            yield intersection

        _start += ONE_DAY


def get_holidays(start, end):
    """
    Возвращает интервалы с выходными днями между `start` и `end`.
    На самом деле, не ограничиваемся справа `end`, а собираем всю
    серию выходных до конца. Если, к примеру, `end` выпадает на субботу,
    то функция вернет, в том числе, и след.воскресенье
    TODO: сейчас захардкожены СБ-ВС в UTC. Нужно брать выходные с календаря
    с учетом часовых поясов участников
    """
    round_to = 24 * 60 * 60  # 24 часа
    start = floor_time(start, round_to)
    end = ceil_time(end, round_to)

    intervals = []
    saturday, sunday = 6, 7

    while start < end:
        if start.isoweekday() in (saturday, sunday):
            intervals.append((start, start + ONE_DAY))
            # Серию выходных собираем до конца
            end += ONE_DAY
        start += ONE_DAY

    return intervals
