# coding: utf-8
"""
Этот модуль и фикстуры no_get_user_tickets, no_get_service_tickets -- копипаста из Фемиды
"""

from __future__ import unicode_literals

from django.conf import settings

from tvm2 import TVM2
from tvm2.protocol import BlackboxClientId


class TVM2Error(Exception):
    pass


def get_tvm_client():
    return TVM2(
        client_id=settings.EASYMEETING_TVM_CLIENT_ID,
        secret=settings.EASYMEETING_TVM_SECRET,
        blackbox_client=BlackboxClientId.ProdYateam,
        destinations=settings.TVM_DESTINATIONS,
    )


def get_service_ticket(client_id):
    """
    Получает один сервисный TVM-тикет
    :param client_id: int
    :return: str
    """
    service_tickets = get_service_tickets(client_id)
    return service_tickets.get(client_id)


def get_service_tickets(*client_ids):
    """
    Получает одним запросом несколько сервисных TVM-тикетов
    :param client_ids: tuple(int)
    :return: {int: str} - словарь client_id: ticket
    """
    client_ids = map(str, client_ids)
    tvm_client = get_tvm_client()
    service_tickets = tvm_client.get_service_tickets(*client_ids)
    service_tickets = {int(k): v for k, v in service_tickets.items()}
    if not list(service_tickets.values()).pop():
        # если хотя бы один тикет не получен - все значения возвращаются None
        raise TVM2Error('Failed to get TVM service tickets')
    return service_tickets


def get_user_ticket(request):
    # Если фронтенд уже авторизовал юзера через tvm2
    ticket_in_request = request.META.get(settings.YAUTH_TVM2_USER_HEADER)
    if ticket_in_request:
        return ticket_in_request

    params = request.yauser.authenticated_by.extract_params(request)
    session_id = params.get('session_id')
    oauth_token = params.get('oauth_token')
    if not session_id and not oauth_token:
        raise TVM2Error('Failed to get session_id or oauth_token from request')

    tvm_client = get_tvm_client()
    user_ticket = tvm_client.get_user_ticket(
        session_id=session_id,
        oauth_token=oauth_token,
        user_ip=params.get('userip'),
        server_host=params.get('server_host'),
    )
    if not user_ticket:
        raise TVM2Error('Failed to get TVM user ticket from blackbox')
    return user_ticket


def _get_user_ticket_by_oauth(oauth_token):
    tvm_client = get_tvm_client()

    user_ticket = tvm_client.get_user_ticket(
        oauth_token=oauth_token,
        user_ip='127.0.0.1',
        server_host='easymeeting.yandex-team.ru',
    )
    if not user_ticket:
        raise TVM2Error('Failed to get TVM user ticket from blackbox')
    return user_ticket


def get_robot_user_ticket():
    return _get_user_ticket_by_oauth(settings.ROBOT_OAUTH_TOKEN)


def get_robot_almighty_user_ticket():
    return _get_user_ticket_by_oauth(settings.ROBOT_ALMIGHTY_OAUTH_TOKEN)
