# coding: utf-8

from __future__ import unicode_literals

import json
import logging

from ylog import context as log_context

from django.views import View as DjangoView
from django.core import exceptions as django_exceptions
from django.utils.functional import cached_property
from django import http

from easymeeting.core.calendar import CalendarController
from easymeeting.lib.tvm2_client import get_user_ticket


log = logging.getLogger(__name__)


class View(DjangoView):
    # можно выключить заворачивание всего в JsonResponse
    response_format = 'json'
    # по умолчанию сериализуем без параметров
    json_response_params_default = {}
    # можно переопределять в своей вьюхе
    json_response_params = {}

    form_cls = None
    form_cls_get = None
    form_cls_post = None

    exceptions_to_handle = (
        django_exceptions.PermissionDenied,
    )

    def dispatch(self, request, *args, **kwargs):
        method_lower = request.method.lower()
        handler = getattr(self, method_lower, None)
        specific_process_handler_exists = hasattr(self, 'process_' + method_lower)
        response_format = kwargs.get('format', self.response_format)

        if handler is not None:
            response = handler(request, *args, **kwargs)
        elif specific_process_handler_exists:
            try:
                response = self.dispatch_process(request, *args, **kwargs)
            except self.exceptions_to_handle as exc:
                response = self.handle_exception(exc, request, args, kwargs)
                if response is None:
                    raise
        else:
            response = self.http_method_not_allowed(request, *args, **kwargs)
        return self.do_response(
            request,
            response,
            response_format=response_format,
        )

    def dispatch_process(self, request, *args, **kwargs):
        method_lower = request.method.lower()
        form = self.maybe_get_form(request)
        process_handler = getattr(self, 'process_' + method_lower)
        if form is not None:
            if not form.is_valid():
                error_response = {
                    'errors': form.errors,
                }
                return self.do_response(
                    request,
                    error_response,
                    response_format='json',
                    status_code=400,
                )
            else:
                data = (
                    getattr(form, 'validated_data')
                    if hasattr(form, 'validated_data')
                    else getattr(form, 'cleaned_data')
                )
                data.update(kwargs)
                with log_context.LogContext(**self.get_logging_context(data)):
                    return process_handler(data=data)
        else:
            data = self.get_query_dict(request)
            data.update(kwargs)
            with log_context.LogContext(**self.get_logging_context(data)):
                return process_handler(data=data)

    def do_response(self, request, response, response_format='json', status_code=200):
        if isinstance(response, dict) and response_format == 'json':
            params = dict(self.json_response_params_default)
            params.update(self.json_response_params)
            return http.JsonResponse(data=response, status=status_code, **params)
        elif isinstance(response, list) and response_format == 'json':
            response = http.JsonResponse(data=response, status=status_code, safe=False)
        return response

    def handle_exception(self, exc, request, args, kwargs):
        if isinstance(exc, django_exceptions.PermissionDenied):
            return self.do_response(
                request=request,
                response={'error': 'PermissionDenied. %s' % str(exc)},
                status_code=403,
            )

    def get_query_dict(self, request):
        if request.method == 'GET':
            return request.GET.copy()
        else:
            if request.content_type is None or request.content_type == 'application/json':
                query_dict = self.get_json_from_request(request)
            else:
                query_dict = request.POST.copy()
            query_dict.update(request.GET.copy())
        return query_dict

    def get_files(self, request):
        return request.FILES

    def get_from_cls(self, request):
        specific_form_cls_name = 'form_cls_' + request.method.lower()
        return getattr(self, specific_form_cls_name) or self.form_cls

    def maybe_get_form(self, request):
        form_cls = self.get_from_cls(request)
        if form_cls is None:
            return
        return form_cls(
            data=self.get_query_dict(request),
        )

    def get_json_from_request(self, request):
        if not hasattr(request, '_json'):
            try:
                request._json = json.load(request)
            except ValueError:
                log.exception('Failed deserialize json')
                request._json = {}
        return request._json

    def get_logging_context(self, data):
        return {}

    @cached_property
    def user_ticket(self):
        return get_user_ticket(self.request)

    @cached_property
    def calendar_controller(self):
        return CalendarController(self.request.yauser.uid, self.user_ticket)
