# coding: utf-8
import logging

from heapq import merge
from itertools import groupby
from operator import itemgetter

from django.db import transaction
from django.utils.functional import cached_property
from django_idm_api.hooks import BaseHooks, AuthHooks
from memoize import memoize

from fb.roles.models import Role, PersonRole, GroupRole
from fb.staff.misc import get_heads_for_persons, get_heads_for_groups
from fb.staff.models import Person, GroupDetails


logger = logging.getLogger(__name__)


class CiaRolesHooks(BaseHooks):

    @cached_property
    def persons_by_id(self):
        persons = Person.objects.filter(
            is_dismissed=False,
            is_robot=False,
        )
        return {p.staff_id: p for p in persons}

    @cached_property
    def groups_by_id(self):
        return {g.staff_id: g for g in GroupDetails.objects.all()}

    def _roles_for_item(self, item, heads):
        values = {h.login: h.get_full_name_idm() for h in heads}
        return {
            'name': item,
            'roles': {
                'slug': 'granter',
                'name': u'Как',
                'values': values,
            },
        }

    def _roles_for_person(self, person, heads):
        return self._roles_for_item(person.get_full_name_idm(), heads)

    def _roles_for_group(self, group, heads):
        return self._roles_for_item(group.name, heads)

    def _volozh_workaround(self, heads_for_person):
        # Set volozh head for volozh
        heads_for_person[2] = [2]

    def _get_subject_roles_values(self):
        heads_for_person = get_heads_for_persons(self.persons_by_id.keys(), all_heads=True)
        self._volozh_workaround(heads_for_person)

        subject_roles_values = {}
        for p_id, head_ids in heads_for_person.iteritems():
            person = self.persons_by_id.get(p_id)
            if person is None:
                continue
            heads = (self.persons_by_id[i] for i in head_ids if i in self.persons_by_id)
            subject_roles_values[person.login] = self._roles_for_person(person, heads)
        return subject_roles_values

    def _get_group_roles_values(self):
        heads_for_group = get_heads_for_groups(self.groups_by_id.keys(), all_heads=True)
        group_roles_values = {}
        for g_id, head_ids in heads_for_group.iteritems():
            group = self.groups_by_id.get(g_id)
            if group is None:
                continue
            heads = (self.persons_by_id[i] for i in head_ids if i in self.persons_by_id)
            group_roles_values[g_id] = self._roles_for_group(group, heads)
        return group_roles_values

    @memoize(1 * 60 * 60)
    def info(self):
        result = {
            'code': 0,
            'roles': {
                'slug': u'type',
                'name': u'Тип',
            },
        }

        group_roles_values = self._get_group_roles_values()
        subject_roles_values = self._get_subject_roles_values()

        subject_roles = {
            'slug': 'subject',
            'name': u'Сотрудник',
            'values': subject_roles_values,
        }

        group_roles = {
            'slug': 'group',
            'name': u'Подразделение',
            'values': group_roles_values,
        }

        result['roles']['values'] = {
            'PersonRole': {
                'name': u'Чтение отзывов о сотруднике',
                'roles': subject_roles,
            },
            'GroupRole': {
                'name': u'Чтение отзывов о подразделении',
                'roles': group_roles,
            },
        }

        return result

    @staticmethod
    def _get_staff_user_by_login(login):
        return Person.objects.get(login=login)

    @staticmethod
    def _get_group_by_id(staff_id):
        return GroupDetails.objects.get(staff_id=staff_id)

    def add_role_impl(self, login, role, fields, **kwargs):
        role_kwargs = {
            'project': role.get('project', Role.PROJECT_FEEDBACK),
            'action': role.get('action', Role.ACTION_READ),
            'granter': self._get_staff_user_by_login(role['granter']),
            'owner': self._get_staff_user_by_login(login),
        }
        role_type = role['type']

        if role_type == 'PersonRole':
            cls = PersonRole
            role_kwargs['subject'] = self._get_staff_user_by_login(role['subject'])
        elif role_type == 'GroupRole':
            cls = GroupRole
            role_kwargs['group'] = self._get_group_by_id(role['group'])
        else:
            raise Exception("Neither `subject` or `group` is presented in fields")

        with transaction.atomic():
            role_create_defaults = {'requester': role_kwargs['granter']}
            role, created = cls.objects.get_or_create(
                defaults=role_create_defaults,
                **role_kwargs
            )
            role.is_approved = True
            role.status = Role.STATUS_OK
            role.save()

        if created:
            role_kwargs.update(role_create_defaults)
            logger.info('Created role with %s', role)
        else:
            logger.info('Approved role with %s', role)

    def remove_role_impl(self, login, role, data, is_fired, **kwargs):
        # Note: у дерева изменилась структура, теперь листовые узлы
        # находятся там, где раньше были промежуточные узлы.
        # Чтобы перенести узлы по-правильному, нужно сначала
        # добавить к листьям unique, засинкать дерево в IDM
        # и уже потом менять структуру, оставляя unique на перемещённых узлах.
        # Но IDM очень плохо справляется с синхронизацией дерева cia,
        # мы не можем идти по такому пути.
        # Т.к. мы просто удаляем узлы без переноса, IDM будет отправлять
        # в систему запросы на удаление ролей. Но роли мы удалять не хотим,
        # поэтому защищаемся таким вот образом.
        # Для новой структуры узел project отсутствует совсем,
        # поэтому там уже всё будет по-честному.
        if role.get('project') == Role.PROJECT_FEEDBACK:
            logger.info('Ignoring role remove request %s', role)
            return

        role_kwargs = {
            'project': role.get('project', Role.PROJECT_FEEDBACK),
            'action': role.get('action', Role.ACTION_READ),
            'owner': self._get_staff_user_by_login(login),
        }
        role_type = role['type']

        if role_type == 'PersonRole':
            cls = PersonRole
            role_kwargs['subject'] = self._get_staff_user_by_login(role['subject'])
        elif role_type == 'GroupRole':
            cls = GroupRole
            role_kwargs['group'] = self._get_group_by_id(role['group'])
        else:
            raise Exception("Type %s is not defined" % role_type)

        with transaction.atomic():
            qs = cls.objects.filter(**role_kwargs)
            count = qs.count()
            qs.delete()
        logger.info('Deleted %d roles matching query %s', int(count), role)

    def get_all_roles_impl(self):
        roles = (
            PersonRole.objects
            .approved()
            .order_by('owner__login')
            .select_related('owner')
        )
        group_roles = (
            GroupRole.objects
            .approved()
            .order_by('owner__login')
            .select_related('owner')
        )
        roles = ((i.owner.login, i) for i in roles)
        group_roles = ((i.owner.login, i) for i in group_roles)
        for login, user_roles in groupby(merge(roles, group_roles), key=itemgetter(0)):
            yield login, [r.as_idm_role() for _, r in user_roles]

    def __repr__(self):
        return self.__class__.__name__


def choose(**kwargs):
    if kwargs.pop('tree', None) == 'cia':
        return CiaRolesHooks(**kwargs)
    return AuthHooks(**kwargs)
