# coding: utf-8
import logging
import os
import re
import uuid
from itertools import chain
try:
    import uwsgi
except ImportError:
    uwsgi = None
from django.contrib.auth.models import AnonymousUser
from django.http import HttpResponse, JsonResponse, HttpResponseForbidden
from django.utils.functional import SimpleLazyObject
from django.core.urlresolvers import reverse
from django.conf import settings
from rest_framework import status
from django_idm_api import conf
from django_yauth.middleware import YandexAuthMiddleware

from fb.roles.models import PersonRole, GroupRole
from fb.staff.models.person import RoleCollection

logger = logging.getLogger(__name__)


class AliveMiddleware(object):
    """
    Простенькая миддлварина, через которую можно проверить,
    что приложение запущено
    """
    ping_re = re.compile(r'^/ping/?$')

    def process_request(self, request):
        if self.ping_re.match(request.path):
            return HttpResponse("I'm alive!")


class ReadOnlyMiddleware(object):
    """
    Миддлварина для перевода сервиса в режим read-only
    """
    def process_request(self, request):
        if request.path.startswith('/admin'):
            return
        if (request.method in ['POST', 'PUT', 'PATCH', 'DELETE']
                and int(os.environ.get('ENABLE_READONLY', 0))):
            return JsonResponse(
                data={'detail': 'Service is in read-only mode.'},
                status=status.HTTP_405_METHOD_NOT_ALLOWED,
            )


class RoleLoader(object):
    """
    Подгружает cia-роли для текущего пользователя:
    1. Пользовательские роли -
      пользователь (owner) может смотреть отзывы на сотрудника (subject),
      будто бы он руководитель сотрудника (granter).
    2. Групповые роли -
      пользователь (owner) может смотреть отзывы сотрудников из группы (group),
      будто бы он руководитель группы (granter)
    """
    def __init__(self, user):
        self.user = user

    def _get_roles(self):
        person_roles = (
            PersonRole.objects
            .approved()
            .filter(
                owner=self.user,
                action=PersonRole.ACTION_READ,
            )
            .select_related('granter', 'subject')
        )
        group_roles = (
            GroupRole.objects
            .approved()
            .filter(
                owner=self.user,
                action=PersonRole.ACTION_READ,
            )
            .select_related('granter', 'group')
        )
        return chain(person_roles, group_roles)

    def __call__(self, *args, **kwargs):
        return RoleCollection(self._get_roles())


class SetRolesFromCIA(object):
    """
    Лениво подгружает cia-роли для текущего пользователя
    """
    def process_request(self, request):
        if request.path == reverse(settings.YAUTH_CREATE_PROFILE_VIEW):
            return

        owner = request.owner
        if owner is None:
            return

        request.owner.roles = SimpleLazyObject(RoleLoader(owner))


class CustomYauthMiddleware(YandexAuthMiddleware):

    def process_request(self, request):
        result = super(CustomYauthMiddleware, self).process_request(request)
        self.safe_user(request)

        if request.path.startswith('/' + conf.IDM_URL_PREFIX) and not self.is_correct_idm_request(request):
            return HttpResponseForbidden('Wrong idm request')

        return result

    def safe_user(self, request):
        try:
            request.user
        except AttributeError:
            request.user = AnonymousUser()

    def is_correct_idm_request(self, request):
        return (
            'idm' in settings.TVM_APPLICATIONS
            and request.yauser.authenticated_by.mechanism_name == 'tvm'
            and request.yauser.service_ticket.src == settings.TVM_APPLICATIONS['idm']
        )


class UwsgiLogMiddleware(YandexAuthMiddleware):

    def process_request(self, request):
        if uwsgi:
            user = getattr(request, 'yauser', None)
            if user:
                try:
                    uwsgi.set_logvar('yandex_login', user.get_username())
                except (AttributeError, NotImplementedError):
                    pass
            request_id = request.META.get("HTTP_X_REQ_ID") or str(uuid.uuid4())
            uwsgi.set_logvar('request_id', request_id)
