import ast
from datetime import datetime
from rest_framework.filters import BaseFilterBackend


class ParamParserMixin(object):
    def __init__(self, *args, **kwargs):
        pass

    @staticmethod
    def _safe_call(value, function):
        if value is None:
            return None
        try:
            value = function(value)
        except StandardError:
            value = None
        return value

    def _get_integer_param(self, request, param_name):
        param = self._get_param(request, param_name)
        param = self._safe_call(param, ast.literal_eval)
        param = self._safe_call(param, int)
        return param

    def _get_integer_params(self, request, param_name):
        params = self._get_param(request, param_name)
        params = self._safe_call(params, ast.literal_eval)

        if not hasattr(params, '__iter__'):
            params = [params]

        params = [self._safe_call(param, int) for param in params]

        if any(param is None for param in params):
            return None
        return params

    @staticmethod
    def _get_param(request, param_name):
        return request.GET.get(param_name, None)

    def _get_bool_param(self, request, param_name):
        param = self._get_param(request, param_name)
        param = self._safe_call(param, ast.literal_eval)
        param = self._safe_call(param, bool)
        return param

    def _get_date_param(self, request, param_name):
        param = self._get_param(request, param_name)
        if param == 'empty':
            return param
        param = self._safe_call(param, lambda x: datetime.strptime(x, "%Y-%m-%d"))
        return param

    def _get_params(self, request, param_name):
        params = self._get_param(request, param_name)
        params = self._safe_call(params, lambda x: x.split(','))
        return params


class BaseBuilder(BaseFilterBackend):

    has_docstring = False

    def filter_queryset(self, request, queryset, view):
        raise NotImplemented

    def __call__(self, *args, **kwargs):
        return self


class FilterMixin(object):
    def _filter_by_field_name(self, queryset, field_name, filter_value):
        if filter_value is None:
            return queryset

        __in = ''
        if isinstance(filter_value, list):
            if len(filter_value) == 1:
                filter_value = filter_value[0]
            else:
                __in = '__in'

        filter_dict = dict()
        filter_dict[field_name + __in] = filter_value
        if getattr(self, 'is_exclude', False):
            return queryset.exclude(**filter_dict)
        return queryset.filter(**filter_dict)


class AttributeQuerysetFilter(ParamParserMixin, FilterMixin, BaseBuilder):

    has_docstring = True

    def __init__(self, *args, **kwargs):
        super(AttributeQuerysetFilter, self).__init__(*args, **kwargs)
        self.attr_name = kwargs.get('attr_name')
        self.query_name = kwargs.get('query_name', self.attr_name)
        self.is_exclude = kwargs.get('exclude', False)

        self.description = 'filter by {0} property'.format(self.attr_name)
        if 'description' in kwargs:
            self.description = kwargs['description']

    def _get_attr_value(self, request):
        raise NotImplemented

    def filter_queryset(self, request, queryset, view):
        value = self._get_attr_value(request)
        return self._filter_by_field_name(queryset, self.query_name, value)

    def __unicode__(self):
        return u'{self.attr_name} -- {self.description}'.format(self=self)


class BoolAttribute(AttributeQuerysetFilter):
    def _get_attr_value(self, request):
        return self._get_bool_param(request, self.attr_name)


class IntegerAttribute(AttributeQuerysetFilter):
    def _get_attr_value(self, request):
        return self._get_integer_param(request, self.attr_name)


class StringAttribute(AttributeQuerysetFilter):
    def _get_attr_value(self, request):
        return self._get_param(request, self.attr_name)


class StringAttributes(AttributeQuerysetFilter):
    def _get_attr_value(self, request):
        return self._get_params(request, self.attr_name)


class IntegerAttributes(AttributeQuerysetFilter):
    def _get_attr_value(self, request):
        return self._get_integer_params(request, self.attr_name)


class Skipper(ParamParserMixin, BaseBuilder):
    def filter_queryset(self, request, queryset, view):
        skip = self._get_integer_param(request, 'skip')
        if skip is not None and skip > 0:
            queryset = queryset[skip:]
        return queryset


class DocstringMetaclass(type):
    def __new__(mcs, class_name, class_parents, class_attr):
        docstring = class_attr.get('__doc__', '')
        if 'filter_backends' in class_attr:
            for _filter in class_attr['filter_backends']:
                if not getattr(_filter, 'has_docstring', False):
                    continue
                if _filter.__doc__:
                    docstring += _filter.__doc__
                else:
                    docstring += unicode(_filter)

                if docstring[-1] != u'\n':
                    docstring += u'\n'

        class_attr['__doc__'] = docstring
        return type(class_name, class_parents, class_attr)
