# coding: utf-8
import requests
import logging

from urlparse import urljoin

from django.conf import settings

logger = logging.getLogger(__name__)


class RelationsAutodetect(object):

    def __init__(self, session=None, **kwargs):
        self._session = session if session is not None else requests.Session()

    def _get_goals_data(self, who, persons):
        goals_base_url = getattr(settings, 'GOALS_BASE_URL')
        goals_oauth_token = getattr(settings, 'FEEDBACK_ROBOT_TOKEN')
        # yandex_internal_root_ca = getattr(settings, 'YANDEX_INTERNAL_ROOT_CA')
        url = urljoin(goals_base_url, '/api/v1/goals/relations/')
        headers = {
            'Authorization': 'OAuth %s' % goals_oauth_token,
        }
        params = {
            'person_login': who.login,
            'related_persons': ','.join([person.login for person in persons])
        }
        try:
            response = self._session.get(
                url,
                params=params,
                headers=headers,
                timeout=2,
                verify=False,
            )
        except requests.RequestException:
            logging.exception('Goals request failed')
            return []
        if not response.ok:
            logger.warning('Goals returned %d status code' % response.status_code)
            return []
        return response.json()

    @staticmethod
    def _process_goals_data(goals_data):
        result = {}
        for person_info in goals_data:
            result[person_info['person_id']] = {
                'as_customer': person_info['as_customer'],
                'as_implementer': person_info['as_implementer'],
            }
        return result

    @staticmethod
    def _get_relation_info(person, relation_type):
        return {
            'person_id': person.staff_id,
            'relation_type': relation_type
        }

    def get_relations(self, who, persons):
        result = []
        person_heads = who.heads
        goals_data = self._get_goals_data(who, persons)
        goals_data = self._process_goals_data(goals_data)
        for person in persons:
            if person.staff_id in person_heads:
                result.append(self._get_relation_info(person, 'subordinate'))
            elif who.staff_id in person.heads:
                result.append(self._get_relation_info(person, 'head'))

            # FIXME: это падает, если _get_goals_data отдаст пустой список. Починить
            elif goals_data[person.staff_id]['as_customer'] > goals_data[person.staff_id]['as_implementer']:
                result.append(self._get_relation_info(person, 'implementer'))
            elif goals_data[person.staff_id]['as_implementer'] > goals_data[person.staff_id]['as_customer']:
                result.append(self._get_relation_info(person, 'client'))

            else:
                result.append(self._get_relation_info(person, 'partner'))
        return result
