from django.utils.decorators import method_decorator
from django.views.decorators.csrf import ensure_csrf_cookie
from django_yauth.util import get_passport_url
from rest_framework import generics
from rest_framework.permissions import AllowAny
from rest_framework.response import Response
from fb.common.serializers import UserInfoSerializer
from fb.staff.serializers import PersonSerializer

from django.db.models import Q
from fb.staff.models import GroupDetails, Person
from collections import OrderedDict


class UserInfoView(generics.GenericAPIView):

    """
        User info view.
    """
    serializer_class = UserInfoSerializer
    permission_classes = (AllowAny,)

    @method_decorator(ensure_csrf_cookie)
    def get(self, request):
        data = dict()
        headers = dict()
        if request.user.is_authenticated():
            staff_user = request.owner
            data = PersonSerializer(staff_user).data
            data['roles'] = request.user.groups.values_list('name', flat=True)
            data['departments'] = staff_user.get_department_ids()
            data['feedback_count'] = staff_user.my_feedback.count()
            data['request_count'] = staff_user.my_requests.count()
            data['active_request_count'] = staff_user.my_requests.filter(
                is_submitted=False,
            ).count()

            data['active_incoming_request_count'] = staff_user.incoming_requests.filter(
                is_submitted=False,
            ).count()

            data['is_head'] = staff_user.groupmembership_set.filter(
                is_chief=True,
            ).exists() or bool(staff_user.roles)
            status_code = 200
        else:
            status_code = 401
            headers['WWW-Authenticate'] = get_passport_url('create')
        return Response(data=data, headers=headers, status=status_code)


class SuggestView(generics.GenericAPIView):

    limit = 5

    def _person_suggest(self, text):
        words = text.split()
        query = Q()
        for word in words:
            query = query & (
                Q(first_name_ru__icontains=word)
                | Q(last_name_ru__icontains=word)
                | Q(first_name_en__icontains=word)
                | Q(last_name_en__icontains=word)
                | Q(login__icontains=word)
            )
        persons = Person.objects.filter(query).exclude(is_dismissed=True)[:self.limit]
        return PersonSerializer(instance=persons, many=True).data

    def _department_suggest(self, text):
        groups = GroupDetails.objects.filter(
            Q(name__icontains=text)
            | Q(name_en__icontains=text)
        )[:self.limit]
        result = []
        for group in groups:
            persons_in_group = group.get_persons()
            serialized_persons = PersonSerializer(instance=persons_in_group, many=True)
            result.append({
                'id': group.staff_id,
                'text': {
                    'ru': group.name,
                    'en': group.name_en,
                },
                'persons': serialized_persons.data,
            })
        return result

    def get(self, request):
        text = unicode(request.GET.get('q', '')).lower()
        types = request.GET.get('types', '')
        if not types:
            types = ['persons', 'departments']

        response = OrderedDict()

        if 'persons' in types:
            response['persons'] = self._person_suggest(text)
        if 'departments' in types:
            response['departments'] = self._department_suggest(text)

        return Response(response)
