# -*- coding: utf-8 -*-
from django.db import models
from django.db.models import signals
from fb.feedback.tasks import send_notification
from django.utils.translation import ugettext_lazy as _


class FeedbackGroup(models.Model):
    person = models.ForeignKey('staff.Person', related_name='feedback_group')
    feedback = models.ForeignKey('feedback.Feedback', related_name='feedback_group')


class BlackList(models.Model):
    person = models.ForeignKey('staff.Person', related_name='blacklist')
    feedback = models.ForeignKey('feedback.Feedback', related_name='blacklist')


class Feedback(models.Model):

    RT_CLIENT = 'client'
    RT_CUSTOMER = 'customer'
    RT_IMPLEMENTER = 'implementer'
    RT_SUBORDINATE = 'subordinate'
    RT_HEAD = 'head'
    RT_PARTNER = 'partner'
    RT_SELF = 'self'

    FM_NEGATIVE = 0
    FM_NEUTRAL = 1
    FM_POSITIVE = 2

    REPORTER_TYPES = (
        (RT_HEAD, _('head')),
        (RT_SUBORDINATE, _('subordinate')),
        (RT_CLIENT, _('client')),
        (RT_CUSTOMER, _('client')),
        (RT_IMPLEMENTER, _('contractor')),
        (RT_PARTNER, _('colleague')),
        (RT_SELF, _('self')),
    )

    REPORTER_TYPES_RU = (
        (RT_CLIENT, u'заказчик'),
        (RT_CUSTOMER, u'заказчик'),
        (RT_IMPLEMENTER, u'исполнитель'),
        (RT_SUBORDINATE, u'подчиненный'),
        (RT_HEAD, u'руководитель'),
        (RT_PARTNER, u'коллега'),
        (RT_SELF, u'отзыв на себя'),
    )

    FEEDBACK_MOODS = (
        (FM_NEGATIVE, _('negative')),
        (FM_NEUTRAL, _('neutral')),
        (FM_POSITIVE, _('positive')),
    )

    reporter = models.ForeignKey('staff.Person', related_name='my_feedback')
    reporter_type = models.CharField(max_length=100, choices=REPORTER_TYPES)
    positive_message = models.TextField(blank=True)
    negative_message = models.TextField(blank=True)
    create_time = models.DateTimeField(auto_now_add=True)
    change_time = models.DateTimeField(auto_now=True)
    is_changed = models.BooleanField(default=False)
    is_revoked = models.BooleanField(default=False)
    revoke_reason = models.TextField(blank=True)

    mood = models.IntegerField(choices=FEEDBACK_MOODS, default=FM_NEUTRAL)

    persons = models.ManyToManyField('staff.Person', through=FeedbackGroup, related_name='feedbacks')
    blacklisted_persons = models.ManyToManyField('staff.Person', through=BlackList, related_name='disabled_feedbacks')

    def get_reporter_type_display_ru(self):
        return dict(self.REPORTER_TYPES_RU).get(self.reporter_type)


class FeedbackRequest(models.Model):
    suggested_persons = models.ManyToManyField('staff.Person', related_name='+')
    suggest_to = models.ForeignKey('staff.Person', related_name='incoming_requests')
    reporter = models.ForeignKey('staff.Person', related_name='my_requests')
    reason = models.TextField(blank=True)
    is_submitted = models.BooleanField(default=False)
    created = models.DateTimeField(auto_now_add=True)
    deadline = models.DateField(null=True, blank=True)


def send_notification_to_heads(sender, **kwargs):
    instance = kwargs.get('instance')
    assert instance is not None
    send_notification.delay(instance.id)


signals.post_save.connect(send_notification_to_heads, sender=Feedback)
