import logging
import operator
import traceback
from datetime import datetime
from collections import defaultdict

from django.core.paginator import Paginator
from django.db.models import Q, Prefetch, Count, Sum
from rest_framework import generics
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework import status
from rest_framework.templatetags.rest_framework import replace_query_param
from rest_framework.settings import api_settings
from fb.feedback.models import Feedback, FeedbackRequest, FeedbackGroup, BlackList
from fb.feedback.permissions import IsOwner, IsOwnerOrSuggestTo, IsOwnerOrHeadChangesMood
from fb.feedback.serializers import (
    FeedbackSerializer,
    FeedbackForReviewSerializer,
    SuggestFeedbackSerializer,
)
from fb.feedback.serializers import RequestFeedbackSerializer, PersonSerializerWithRaiting
from fb.staff.models import Person, Group, GroupMembership
from fb.renderers import FlatCSVRenderer
from fb.staff.serializers import PersonSerializer, SubordinatesPersonSerializer

from fb.common.filter_backends import ParamParserMixin, IntegerAttribute
from fb.common.permissions import OAuthClientPermission

from fb.feedback.tasks import ask_feedback, send_feedback_request_reject

from fb.quizz.models import PersonAnswer, QuizzQuestion
from fb.quizz.serializers import QuizzQuestionWithPersonAnswerSerializer, PersonAnswerSerializer

log = logging.getLogger(__name__)


class FeedbackList(generics.ListCreateAPIView):
    serializer_class = FeedbackSerializer

    def pre_save(self, obj):
        staff_user = Person.objects.get(uid=self.request.yauser.uid)
        obj.reporter_id = staff_user.staff_id

    def get_queryset(self):
        staff_user = Person.objects.get(uid=self.request.yauser.uid)
        queryset = Feedback.objects.filter(
            reporter=staff_user,
        ).exclude(
            Q(positive_message='') & Q(negative_message='')
        ).select_related(
            'reporter',
        ).prefetch_related(
            'persons',
            'blacklisted_persons',
            'personanswer_set__person',
            'personanswer_set__on_whom',
            'personanswer_set__quizz_answer',
            'personanswer_set__quizz_answer__question',
        ).order_by('-create_time')
        return queryset.all()

    def create(self, request, *args, **kwargs):
        serializer = self.get_serializer(data=request.DATA, files=request.FILES)

        if serializer.is_valid():
            self.pre_save(serializer.object)
            self.object = serializer.save(force_insert=True)
            self.post_save(self.object, created=True)
            request_id = request.DATA.get('request_id', None)
            if request_id is not None:
                FeedbackRequest.objects.filter(
                    id=request_id,
                    suggest_to=request.owner,
                ).update(is_submitted=True)
            headers = self.get_success_headers(serializer.data)
            return Response(serializer.data, status=status.HTTP_201_CREATED,
                            headers=headers)

        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    def get(self, request):
        return Response(FeedbackSerializer(
            self.get_queryset(),
            many=True,
            context={'requester': request.owner}
        ).data)


class FeedbackDetails(generics.RetrieveUpdateAPIView):
    model = Feedback
    serializer_class = FeedbackSerializer
    lookup_field = 'id'
    permission_classes = (
        IsAuthenticated,
        OAuthClientPermission,
        IsOwnerOrHeadChangesMood,
    )

    def pre_save(self, obj):
        super(FeedbackDetails, self).pre_save(obj)
        obj.is_changed = True


class FilterFeedbacksMixin(object):

    def _q_for_controlled_departments(self, paths):
        group_condition = reduce(operator.or_, (
            Q(group_id__in=Group.objects.filter(path__startswith=path))
            for path in paths
        ), Q()) & Q(is_primary=True)

        return GroupMembership.objects.filter(
            group_condition,
        ).values_list('person_id', flat=True)

    def _filter(self, request_owner, departments):
        paths = Group.objects.filter(
            staff_id__in=departments,
        ).values_list('path', flat=True)

        lca_paths = set([])
        for p1 in paths:
            for p2 in paths:
                if p1.startswith(p2) and p1 != p2:
                    break
            else:
                lca_paths.add(p1)

        feedback_ids = FeedbackGroup.objects

        if lca_paths:
            subject_ids = self._q_for_controlled_departments(lca_paths)

            feedback_ids = feedback_ids.filter(
                person_id__in=subject_ids,
            ).exclude(
                person=request_owner,
            )

        else:
            subject_ids = []
            feedback_ids = feedback_ids.none()
        return subject_ids, feedback_ids

    def _get_queryset(self):
        request_owner = self.request.owner

        departments = request_owner.get_controlled_departments()

        subject_ids, feedback_ids = self._filter(request_owner, departments)

        feedback_groups = FeedbackGroup.objects
        person_prefetch_filter = Person.objects.filter(
            staff_id__in=subject_ids,
        )

        for role in request_owner.roles:
            if role.is_group_role:
                _s_ids, _f_ids = self._filter(
                    request_owner, [role.group.staff_id],
                )

                feedback_ids = (
                    feedback_ids | _f_ids.exclude(
                        person=role.granter,
                    )
                )

                person_prefetch_filter = person_prefetch_filter | Person.objects.filter(
                    staff_id__in=_s_ids,
                )

            elif role.is_person_role:
                if role.subject == request_owner:
                    continue
                feedback_ids = feedback_ids | feedback_groups.filter(person=role.subject)
                person_prefetch_filter = person_prefetch_filter | Person.objects.filter(
                    staff_id=role.subject.staff_id,
                )
            else:
                continue
        return feedback_ids, person_prefetch_filter

    def get_feedbacks_queryset(self, person_id):
        feedback_ids, person_prefetch_filter = self._get_queryset()
        if person_id is not None:
            feedback_ids = feedback_ids.filter(person=person_id)

        feedback_ids = (
            feedback_ids.values_list('feedback', flat=True).distinct()
        )

        blacklisted_feedbacks = (
            BlackList.objects.filter(
                person=self.request.owner,
            ).values_list('feedback', flat=True).distinct()
        )

        queryset = Feedback.objects.filter(
            id__in=feedback_ids,
        ).exclude(
            id__in=blacklisted_feedbacks,
        ).prefetch_related(
            'blacklisted_persons',
            'personanswer_set__on_whom',
            'personanswer_set__person',
            'personanswer_set__quizz_answer',
            Prefetch('persons', queryset=person_prefetch_filter)
        ).select_related(
            'reporter',
        ).order_by('-create_time')

        return queryset.all()


class NativeViewPaginator(object):

    def get_page_number(self):
        return self.request.QUERY_PARAMS.get(self.page_query_param, 1)

    def get_page_size(self):
        return self.request.QUERY_PARAMS.get(self.pagiante_by_param, self.paginate_by)

    def get_next_link(self, page):
        if not page.has_next():
            return None
        url = self.request.build_absolute_uri()
        page_number = page.next_page_number()
        return replace_query_param(url, self.page_query_param, page_number)

    def get_prev_link(self, page):
        if not page.has_previous():
            return None
        url = self.request.build_absolute_uri()
        page_number = page.previous_page_number()
        return replace_query_param(url, self.page_query_param, page_number)

    def paginate_queryset(self, queryset):
        page_size = self.get_page_size()
        paginated = Paginator(queryset, page_size)
        return paginated

    def wrap_data_with_page_context(self, data, paginated_queryset):
        page = paginated_queryset.page(self.get_page_number())
        return {
            'count': paginated_queryset.count,
            'next': self.get_next_link(page),
            'prev': self.get_prev_link(page),
            'results': data,
        }


class MySubordinatesDepartmentFeedbackList(FilterFeedbacksMixin, generics.ListAPIView):
    serializer_class = FeedbackSerializer

    def pre_save(self, obj):
        staff_user = Person.objects.get(uid=self.request.yauser.uid)
        obj.reporter_id = staff_user.staff_id

    def filter_feedback_queryset(self, queryset):
        param_parser = ParamParserMixin()
        from_date = param_parser._get_date_param(self.request, 'from_date')
        if from_date is not None:
            queryset = queryset.filter(create_time__gte=from_date)
        to_date = param_parser._get_date_param(self.request, 'to_date')
        if to_date is not None:
            queryset = queryset.filter(create_time__lte=to_date)
        reporter_type = param_parser._get_param(self.request, 'reporter_type')
        if reporter_type is not None:
            queryset = queryset.filter(reporter_type=reporter_type)
        return queryset

    def filter_request_queryset(self, queryset):
        param_parser = ParamParserMixin()
        from_date = param_parser._get_date_param(self.request, 'from_date')
        if from_date is not None:
            queryset = queryset.filter(created__gte=from_date)
        to_date = param_parser._get_date_param(self.request, 'to_date')
        if to_date is not None:
            queryset = queryset.filter(created__lte=to_date)
        return queryset

    def get_answers_queryset(self, person_id, feedback_queryset):
        queryset = PersonAnswer.objects.filter(
            on_whom_id=person_id,
            feedback__in=feedback_queryset,
        ).select_related(
            'person',
            'on_whom',
        )
        return queryset

    def get_quizz_questions_queryset(self, quizz_answers_queryset):
        question_ids = quizz_answers_queryset.values_list('quizz_answer__question', flat=True).distinct()
        queryset = QuizzQuestion.objects.filter(
            pk__in=question_ids,
        ).prefetch_related(
            'quizzquestionanswer_set',
            'quizzquestionanswer_set__question',
        )
        return queryset

    def get_paginate_by(self, queryset=None):
        if 'page_size' in self.request.QUERY_PARAMS:
            return 50
        return None

    def get(self, request, pk=None, login=None):
        try:
            if pk:
                person_filter = {'pk': pk}
            else:
                person_filter = {'login': login}
            person = (
                Person.objects
                .prefetch_related('groupmembership_set')
                .get(**person_filter)
            )
            if not pk:
                pk = person.pk
        except Person.DoesNotExist:
            return Response(status=404)
        requester = self.request.owner
        if not requester.is_head_of(person) and not requester.has_read_role(person):
            return Response({"reason": "You have no permissions"}, 403)

        feedback_queryset = self.get_feedbacks_queryset(pk)
        feedback_queryset = self.filter_feedback_queryset(feedback_queryset)
        quizz_answers_to_person = self.get_answers_queryset(pk, feedback_queryset)
        quizz_questions = self.get_quizz_questions_queryset(quizz_answers_to_person)
        person_requests = (
            FeedbackRequest.objects
            .filter(suggested_persons=person)
            .prefetch_related('suggested_persons')
            .select_related('suggest_to', 'reporter')
            .order_by('-created')
        )
        person_requests = self.filter_request_queryset(person_requests)

        not_submitted_person_requests = person_requests.filter(
            is_submitted=False,
        ).all()

        submitted_person_requests = person_requests.filter(
            is_submitted=True,
        ).all()

        serialized_person = SubordinatesPersonSerializer(instance=person).data

        serialized_feedbacks = FeedbackSerializer(
            instance=feedback_queryset,
            many=True,
            context={'requester': requester},
        ).data

        serialized_quizz_questions = QuizzQuestionWithPersonAnswerSerializer(
            instance=quizz_questions,
            many=True,
            context={
                'person': person,
                'filtered_feedback_ids': [
                    feedback.id for feedback in feedback_queryset
                ]
            },
        ).data

        serialized_person_requests = RequestFeedbackSerializer(
            instance=not_submitted_person_requests,
            many=True,
        ).data

        return Response({
            'person': serialized_person,
            'requests': serialized_person_requests,
            'quizz_questions': serialized_quizz_questions,
            'feedbacks': serialized_feedbacks,
            'submitted_requests_count': submitted_person_requests.count(),
            'not_submitted_requests_count': not_submitted_person_requests.count(),
        })


class FeedbackListForReviewService(generics.ListAPIView):

    def get(self, request, pk=None, login=None):
        try:
            if pk:
                person_filter = {'pk': pk}
            elif login:
                person_filter = {'login': login}
            person = Person.objects.get(**person_filter)
        except Person.DoesNotExist:
            return Response(status=404)
        is_review_robot = request.user.username == 'robot-review'
        if not is_review_robot:
            return Response({'reason': 'You have no permissions'}, 403)

        feedbacks = Feedback.objects.filter(
            feedback_group__person=person
        ).select_related('reporter').order_by('-create_time')

        params = self.request.GET
        from_date = params.get('from_date')
        if from_date is not None:
            feedbacks = feedbacks.filter(create_time__gte=from_date)
        to_date = params.get('to_date')
        if to_date is not None:
            feedbacks = feedbacks.filter(create_time__lte=to_date)
        reporter_type = params.get('reporter_type')
        if reporter_type is not None:
            feedbacks = feedbacks.filter(reporter_type=reporter_type)

        serialized_feedbacks = FeedbackForReviewSerializer(
            instance=feedbacks,
            many=True,
        ).data
        return Response(dict(
            feedbacks=serialized_feedbacks,
            quizz_questions=self._get_quizzes_json_safe(person, feedbacks)
        ))

    def _get_quizzes_json_safe(self, person, feedback_q):
        try:
            return self._get_quizzes_json(person, feedback_q)
        except Exception:
            log.warning("Can't get quizze answers for {} due to {}".format(person.login, traceback.format_exc()))
            return []

    def _get_quizzes_json(self, person, feedback_q):
        # this and internal functions is just a copypaste from MySubordinatesDepartmentFeedbackList
        quizz_answers_to_person = self._get_answers_queryset(person.pk, feedback_q)
        quizz_questions = self._get_quizz_questions_queryset(quizz_answers_to_person)

        serialized_quizz_questions = QuizzQuestionWithPersonAnswerSerializer(
            instance=quizz_questions,
            many=True,
            context={
                'person': person,
                'filtered_feedback_ids': [
                    feedback.id for feedback in feedback_q
                ]
            },
        ).data
        return serialized_quizz_questions

    def _get_answers_queryset(self, person_id, feedback_queryset):
        queryset = PersonAnswer.objects.filter(
            on_whom_id=person_id,
            feedback__in=feedback_queryset,
        ).select_related(
            'person',
            'on_whom',
        )
        return queryset

    def _get_quizz_questions_queryset(self, quizz_answers_queryset):
        question_ids = quizz_answers_queryset.values_list('quizz_answer__question', flat=True).distinct()
        queryset = QuizzQuestion.objects.filter(
            pk__in=question_ids,
        ).prefetch_related(
            'quizzquestionanswer_set',
            'quizzquestionanswer_set__question',
        )
        return queryset


class MySubordinatesDepartmentFeedbackPersonsList(NativeViewPaginator, FilterFeedbacksMixin, generics.ListAPIView):
    serializer_class = PersonSerializer
    renderer_classes = api_settings.DEFAULT_RENDERER_CLASSES + [FlatCSVRenderer, ]
    paginate_by = 50
    pagiante_by_param = 'page_size'
    page_query_param = 'page'
    filter_backends = [
        IntegerAttribute(attr_name='person_id', query_name='staff_id'),
    ]

    def filter_feedback_queryset(self, queryset):
        param_parser = ParamParserMixin()
        from_date = param_parser._get_date_param(self.request, 'from_date')
        if from_date is not None:
            queryset = queryset.filter(create_time__gte=from_date)
        to_date = param_parser._get_date_param(self.request, 'to_date')
        if to_date is not None:
            queryset = queryset.filter(create_time__lte=to_date)
        reporter_type = param_parser._get_param(self.request, 'reporter_type')
        if reporter_type is not None:
            queryset = queryset.filter(reporter_type=reporter_type)
        return queryset

    def filter_feedback_group_queryset(self, queryset):
        param_parser = ParamParserMixin()
        from_date = param_parser._get_date_param(self.request, 'from_date')
        if from_date is not None:
            queryset = queryset.filter(feedback__create_time__gte=from_date)
        to_date = param_parser._get_date_param(self.request, 'to_date')
        if to_date is not None:
            queryset = queryset.filter(feedback__create_time__lte=to_date)
        reporter_type = param_parser._get_param(self.request, 'reporter_type')
        if reporter_type is not None:
            queryset = queryset.filter(feedback__reporter_type=reporter_type)
        return queryset

    def filter_request_queryset(self, queryset):
        param_parser = ParamParserMixin()
        from_date = param_parser._get_date_param(self.request, 'from_date')
        if from_date is not None:
            queryset = queryset.filter(created__gte=from_date)
        to_date = param_parser._get_date_param(self.request, 'to_date')
        if to_date is not None:
            queryset = queryset.filter(created__lte=to_date)
        return queryset

    def get_queryset(self):
        feedback_ids, _ = self._get_queryset()
        feedback_ids = self.filter_feedback_group_queryset(feedback_ids)

        persons_ids = (
            feedback_ids.values_list('person_id', flat=True).distinct()
        )

        queryset = Person.objects.filter(staff_id__in=persons_ids)
        param_parser = ParamParserMixin()
        group_id = param_parser._get_integer_param(self.request, 'group_id')
        if group_id is not None:
            try:
                group = Group.objects.filter(staff_id=group_id).get()
            except Group.DoesNotExist:
                queryset = queryset.none()
            else:
                queryset = queryset.filter(
                    groups__group_structure__path__startswith=group.path,
                )
        return self.filter_queryset(queryset.all())

    def get_feedback_moods(self, feedbacks):
        moods = feedbacks.values('mood').annotate(count=Count('mood')).order_by()
        moods = defaultdict(int, [(mood['mood'], mood['count']) for mood in moods])
        feedback_stats = {
            'positive_count': moods[Feedback.FM_POSITIVE],
            'neutral_count': moods[Feedback.FM_NEUTRAL],
            'negative_count': moods[Feedback.FM_NEGATIVE],
        }
        return feedback_stats

    def get_requests_statuses(self, user):
        feedback_requests_statuses = self.filter_request_queryset(FeedbackRequest.objects)
        feedback_requests_statuses = (
            feedback_requests_statuses
            .filter(suggested_persons=user)
            .values('is_submitted')
            .annotate(count=Count('is_submitted'))
        )
        submitted_requests_count = 0
        not_submitted_requests_count = 0
        for feedback_requests_status in feedback_requests_statuses:
            if feedback_requests_status['is_submitted']:
                submitted_requests_count = feedback_requests_status['count']
            else:
                not_submitted_requests_count = feedback_requests_status['count']
        requests_stats = {
            'count': submitted_requests_count + not_submitted_requests_count,
            'applied_count': submitted_requests_count,
        }
        return requests_stats

    def get_competences_average_values(self, user, feedbacks):
        competention_answers = PersonAnswer.objects.filter(
            feedback__in=feedbacks.filter(is_revoked=False).all(),
            on_whom=user,
            quizz_answer__question__question_type=QuizzQuestion.QT_COMPETENCE,
        ).exclude(
            quizz_answer__is_open=True,
        ).select_related(
            'quizz_answer',
            'quizz_answer__question',
        ).values(
            'quizz_answer__question__text',
        ).order_by(
            'quizz_answer__question__text',
        ).annotate(
            sum=Sum('quizz_answer__text'),
            count=Count('quizz_answer__id'),
        )

        aggregated = {}
        for c in competention_answers:
            title = c['quizz_answer__question__text']
            a = aggregated.setdefault(title, {'title': title, 'sum': 0, 'count': 0})
            a['sum'] += c['sum']
            a['count'] += c['count']

        competences = map(
            lambda x: {
                'title': x['title'],
                'average_value': round(float(x['sum']) / x['count'], 2),
            },
            aggregated.itervalues()
        )
        return competences

    def _is_format_request(self, format_):
        query_params = getattr(self.request, 'QUERY_PARAMS', None)
        return query_params and query_params.get('format') == format_

    def _is_csv_request(self):
        return self._is_format_request('csv')

    def get(self, request, *args, **kwargs):
        queryset = self.get_queryset().prefetch_related('feedbacks')
        if self._is_csv_request():
            users = queryset
        else:
            paginated_queryset = self.paginate_queryset(queryset)
            users = paginated_queryset.page(self.get_page_number()).object_list
        result = []
        for user in users:
            serialized_person = PersonSerializer(instance=user).data

            user_feedbacks = self.get_feedbacks_queryset(user.staff_id)
            user_feedbacks = self.filter_feedback_queryset(user_feedbacks)

            feedback_stats = self.get_feedback_moods(user_feedbacks)
            requests_stats = self.get_requests_statuses(user)
            competences = self.get_competences_average_values(user, user_feedbacks)

            if self._is_csv_request():
                result.append({
                    'person': user.get_full_name(),
                    'positive_feedback_count': feedback_stats['positive_count'],
                    'neutral_feedback_count': feedback_stats['neutral_count'],
                    'negative_feedback_count': feedback_stats['negative_count'],
                    'requests_count': requests_stats['count'],
                    'applied_requests_count': requests_stats['applied_count'],
                })
            else:
                result.append({
                    'person': serialized_person,
                    'feedback_stats': feedback_stats,
                    'requests_stats': requests_stats,
                    'competences': competences,
                })
        if self._is_csv_request():
            response = Response(result)
        else:
            response = Response(self.wrap_data_with_page_context(result, paginated_queryset))
        if self._is_csv_request():
            response['Content-Disposition'] = (
                'attachment; filename="feedback-{time}.csv"'.format(
                    time=datetime.now().strftime('%Y-%m-%d'),
                )
            )
        return response


class DepartmentFeedbackList(FilterFeedbacksMixin, generics.ListAPIView):
    serializer_class = FeedbackSerializer

    def pre_save(self, obj):
        staff_user = Person.objects.get(uid=self.request.yauser.uid)
        obj.reporter_id = staff_user.staff_id

    def get_queryset(self):
        person_id = self.request.QUERY_PARAMS.get('person_id', None)

        feedback_ids, person_prefetch_filter = self._get_queryset()
        if person_id is not None:
            feedback_ids = feedback_ids.filter(person=person_id)

        feedback_ids = (
            feedback_ids.values_list('feedback', flat=True).distinct()
        )

        blacklisted_feedbacks = (
            BlackList.objects.filter(
                person=self.request.owner,
            ).values_list('feedback', flat=True).distinct()
        )

        queryset = Feedback.objects.filter(
            id__in=feedback_ids,
        ).exclude(
            id__in=blacklisted_feedbacks,
        ).prefetch_related(
            'blacklisted_persons',
            'personanswer_set__on_whom',
            'personanswer_set__person',
            'personanswer_set__quizz_answer',
            Prefetch('persons', queryset=person_prefetch_filter)
        ).select_related(
            'reporter',
        ).order_by('-create_time')

        # CIA-44
        param_parser = ParamParserMixin()
        from_date = param_parser._get_date_param(self.request, 'from_date')
        if from_date is not None:
            queryset = queryset.filter(create_time__gte=from_date)
        to_date = param_parser._get_date_param(self.request, 'to_date')
        if to_date is not None:
            queryset = queryset.filter(create_time__lte=to_date)

        return queryset

    def get_serializer_context(self):
        context = super(DepartmentFeedbackList, self).get_serializer_context()
        if hasattr(self.request, 'owner'):
            context['requester'] = self.request.owner
        return context

    def get_paginate_by(self, queryset=None):
        if 'page_size' in self.request.QUERY_PARAMS:
            return 50
        return None


class DepartmentFeedbackPersonsList(FilterFeedbacksMixin, generics.ListAPIView):
    serializer_class = PersonSerializerWithRaiting

    def get_queryset(self):
        feedback_ids, _ = self._get_queryset()
        persons_ids = (
            feedback_ids.values_list('person_id', flat=True).distinct()
        )

        queryset = Person.objects.filter(staff_id__in=persons_ids)
        return queryset.all()


class RequestFeedback(generics.ListAPIView):
    serializer_class = RequestFeedbackSerializer
    queryset = Feedback.objects.none()

    def get_queryset(self):
        staff_user = Person.objects.get(uid=self.request.yauser.uid)
        queryset = FeedbackRequest.objects.filter(
            reporter=staff_user,
            is_submitted=False,
        ).prefetch_related(
            'suggested_persons',
        ).select_related(
            'suggest_to',
            'reporter',
        )
        return queryset.all()

    def post(self, request):
        log.info("Started request serialization")
        serializer = SuggestFeedbackSerializer(data=request.DATA, files=request.FILES)
        try:
            staff_user = Person.objects.get(uid=self.request.yauser.uid)
        except Person.DoesNotExist:
            log.warning("Person with uid: {} does not exist".format(self.request.yauser.uid))
            return Response(status=403)

        if serializer.is_valid():
            suggested_persons = serializer.data['suggested_persons'] or []
            persons_ids = list(p['id'] for p in suggested_persons)
            reporters = serializer.data['suggest_to'] or []
            reporters_ids = map(lambda x: x['id'], reporters)
            reason = serializer.data['reason']
            is_group_request = serializer.data['is_group_request']
            deadline = serializer.data['deadline']
            request_ids = []
            created = []
            for suggest_to in reporters_ids:
                if is_group_request:
                    request = self.create_request(reason, staff_user, suggest_to, persons_ids, deadline)
                    created.append({
                        'id': request.id,
                        'reporter': suggest_to,
                        'suggested_persons': persons_ids,
                    })
                    request_ids.append(request.id)
                else:
                    for person_id in persons_ids:
                        request = self.create_request(reason, staff_user, suggest_to, [person_id], deadline)
                        created.append({
                            'id': request.id,
                            'reporter': suggest_to,
                            'suggested_persons': [person_id],
                        })
                        log.info("Add  celery task: ask_feedback: "
                                 "staff_id: {}, person_id: {}".format(staff_user.staff_id,
                                                                      person_id)
                                 )
                        ask_feedback.delay(staff_user.staff_id, [person_id], reason, [request.id])
            if is_group_request:
                log.info("Add  celery task: ask_feedback to group request: "
                         "staff_id: {}, person_ids: {}".format(staff_user.staff_id,
                                                               persons_ids)
                         )
                ask_feedback.delay(staff_user.staff_id, persons_ids, reason, request_ids)
            log.info("Started response serialization")
            response = serializer.data.copy()
            response['created'] = created
            log.info("Ended response serialization")
            return Response(status=200, data=response)
        log.error("serializer is not valid errors: {}".format(serializer.errors))
        return Response(status=400, data=serializer.errors)

    def create_request(self,
                       reason, reporter, suggest_to, suggested_persons, deadline=None):
        request = FeedbackRequest(
            reason=reason,
            reporter=reporter,
            suggest_to_id=suggest_to,
            deadline=deadline,
        )
        request.save()
        request.suggested_persons.add(
            *Person.objects.filter(staff_id__in=suggested_persons).all()
        )
        return request


class IncomingRequestList(generics.ListAPIView):
    serializer_class = RequestFeedbackSerializer
    queryset = Feedback.objects.none()

    def get_queryset(self):
        staff_user = Person.objects.get(uid=self.request.yauser.uid)
        queryset = FeedbackRequest.objects.filter(
            suggest_to=staff_user,
            is_submitted=False,
        ).prefetch_related(
            'suggested_persons',
        ).select_related(
            'suggest_to',
            'reporter',
        )
        return queryset.all()


class RequestList(generics.ListCreateAPIView):
    serializer_class = FeedbackSerializer

    def pre_save(self, obj):
        staff_user = Person.objects.get(uid=self.request.yauser.uid)
        obj.reporter_id = staff_user.staff_id

    def get_queryset(self):
        staff_user = Person.objects.get(uid=self.request.yauser.uid)
        queryset = Feedback.objects.filter(reporter=staff_user)
        return queryset.all()


class RequestDetails(generics.RetrieveUpdateAPIView):
    model = FeedbackRequest
    serializer_class = RequestFeedbackSerializer
    lookup_field = 'id'
    permission_classes = (
        IsAuthenticated,
        OAuthClientPermission,
        IsOwnerOrSuggestTo,
    )

    def partial_update(self, request, *args, **kwargs):
        response = super(RequestDetails, self).partial_update(
            request, *args, **kwargs)
        self._handle_possible_reject(request, *args, **kwargs)
        return response

    def _handle_possible_reject(self, request, *args, **kwargs):
        if not request.DATA.get('is_rejected'):
            return

        feedback_request = self.get_object()
        if feedback_request.is_submitted:
            return
        feedback_request.is_submitted = True
        feedback_request.save()

        send_feedback_request_reject.delay(
            request_id=feedback_request.pk
        )


class AskRequest(generics.RetrieveUpdateAPIView):
    model = FeedbackRequest
    serializer_class = RequestFeedbackSerializer
    lookup_field = 'id'
    permission_classes = (
        IsAuthenticated,
        OAuthClientPermission,
        IsOwner,
    )

    def partial_update(self, request, *args, **kwargs):
        return self.update(request, *args, **kwargs)

    def update(self, request, *args, **kwargs):
        self.object = self.get_object()
        serializer = self.get_serializer(self.object)
        ask_feedback.delay(
            self.object.reporter.staff_id,
            list(self.object.suggested_persons.values_list('staff_id', flat=True)),
            self.object.reason,
            [self.object.id],
        )
        return Response(serializer.data)
