from django.db import models
from treebeard.mp_tree import MP_Node

from fb.staff.models.person import Person

__all__ = ['Group', 'GroupDetails']


class Group(MP_Node):

    class Meta:
        app_label = 'staff'
        db_table = 'staff_group'

    staff = models.OneToOneField(
        'staff.GroupDetails',
        to_field='staff_id',
        db_column='staff_id',
        on_delete=models.DO_NOTHING,
        related_name='group_structure',
    )

    @classmethod
    def _get_ancestors_pathes(cls, path):
        for pos in range(0, len(path), cls.steplen)[1:]:
            yield path[0:pos]
        yield path


class GroupDetails(models.Model):

    SEARCH_HEAD_DEPARTMENTS = [
        64, 8150, 1613, 29, 65, 139, 236, 47, 1680,
        1618, 24768, 24741, 24735, 29524, 27036, 112, 288,
        8676, 1619, 31, 242, 24881, 24693, 1615,
    ]

    YANDEX_HEAD_DEPARTMENTS = [
        962, 5, 1110, 29314,
        1558, 8, 1561, 52, 1577, 13, 1564, 142, 1543, 26555, 19, 1563, 29452,
        24701, 922, 21, 22, 9175, 24683, 9, 51,
        1106, 8174,
    ]

    ALL_HEAD_DEPARTMENTS = SEARCH_HEAD_DEPARTMENTS + YANDEX_HEAD_DEPARTMENTS

    STATS_ENABLED = [
        ['search', 1558],
        ['exploitation', 8],
        ['distproducts', 1561],
        ['content', 52],
    ]

    class Meta:
        app_label = 'staff'
        db_table = 'staff_group_info'
        verbose_name_plural = 'Group details'

    name = models.CharField(max_length=255)
    name_en = models.CharField(max_length=255, blank=True, default="")
    staff_id = models.IntegerField(unique=True, primary_key=True)
    machine_name = models.CharField(max_length=255, default="")

    def get_ancestors(self, include_self=False):
        ancestors = self.get_ancestors_ids(include_self)
        return GroupDetails.objects.filter(
            staff_id__in=ancestors,
        ).order_by('group_structure__path').all()

    def get_ancestors_ids(self, include_self=False):
        ancestors = self.group_structure.get_ancestors().values_list(
            'staff_id', flat=True)
        if include_self:
            ancestors = list(ancestors)
            ancestors.append(self.staff_id)
        return ancestors

    def get_heads_list(self):
        for membership in self.groupmembership_set.all():
            if membership.is_chief or membership.is_deputy:
                yield membership.person

    def get_chief(self):
        for membership in self.groupmembership_set.all():
            if membership.is_chief:
                return membership.person

    def get_heads_dict(self):
        return [
            {'id': membership.person_id}
            for membership in self.groupmembership_set.all()
            if membership.is_chief or membership.is_deputy
        ]

    def get_persons(self):
        return Person.objects.filter(
            pk__in=self.groupmembership_set.values_list('person_id'),
            is_dismissed=False,
        )

    @property
    def name_i18n(self):
        return {
            'ru': self.name,
            'en': self.name_en,
        }

    @property
    def url(self):
        return 'https://staff.yandex-team.ru/departments/{self.machine_name}/'.format(self=self)

    def __unicode__(self):
        return self.name
