# coding: utf-8
import operator
from itertools import chain
from django.db import models
from django.db.models import Q
from fb.staff.misc import get_heads_for_persons
from fb.staff.misc.inflector import get_inflator
from django.utils import translation


import logging
log = logging.getLogger(__name__)

__all__ = ['Person']


class Person(models.Model):

    volozh_staff_id = 2
    bunina_staff_id = 1047

    PG_MALE = 0
    PG_FEMALE = 1

    PERSON_GENDERS = (
        (PG_MALE, 'male'),
        (PG_FEMALE, 'female'),
    )

    class Meta:
        app_label = 'staff'
        db_table = 'staff_person'

    staff_id = models.IntegerField(unique=True, primary_key=True)
    uid = models.BigIntegerField(db_index=True)
    login = models.CharField(max_length=50)

    is_dismissed = models.BooleanField(default=False)
    gender = models.IntegerField(choices=PERSON_GENDERS, default=PG_MALE)

    first_name_ru = models.CharField(max_length=50, blank=True)
    last_name_ru = models.CharField(max_length=50, blank=True)

    first_name_en = models.CharField(max_length=50, blank=True)
    last_name_en = models.CharField(max_length=50, blank=True)

    city_name_ru = models.TextField(blank=True, default="")
    city_name_en = models.TextField(blank=True, default="")

    official_position_ru = models.TextField(blank=True, default="")
    official_position_en = models.TextField(blank=True, default="")

    language_ui = models.CharField(max_length=10, blank=True, default="ru")

    official_join_at = models.DateField(null=True, default=None)

    groups = models.ManyToManyField('staff.GroupDetails',
                                    through='staff.GroupMembership',
                                    related_name='members')

    is_robot = models.NullBooleanField(default=False)

    def __init__(self, *args, **kwargs):
        self._heads = None
        self._head = None
        self.hr_head_departments = []
        self.hr_departments = []
        self._first_name_cases = None
        self._last_name_cases = None
        self.roles = RoleCollection([])
        super(Person, self).__init__(*args, **kwargs)

    def get_full_name_ru(self):
        return u'{} {}'.format(self.first_name_ru, self.last_name_ru)

    def get_full_name_en(self):
        return u'{} {}'.format(self.first_name_en, self.last_name_en)

    def get_full_name(self):
        if translation.get_language() == 'ru':
            if self.first_name_ru and self.last_name_ru:
                return self.get_full_name_ru()
        return self.get_full_name_en()

    def get_full_name_idm(self):
        return (
            self.get_full_name_ru().strip()
            or self.get_full_name_en().strip()
            or self.login
        )

    def set_name_cases(self):
        if self._first_name_cases is None or self._last_name_cases is None:
            # кажется, это нигде не используется, но ходится каждый раз и без кэша
            cases = ['nom', 'gen', 'dat', 'acc', 'ins', 'abl']
            self._first_name_cases = {
                case: self.first_name_ru
                for case in cases
            }
            self._last_name_cases = {
                case: self.last_name_ru
                for case in cases
            }
            # inflator = get_inflator()
            # inflator.set_cases_for_person(self)

    def get_first(self):
        """ First names for different languages. """
        return {
            'en': self.first_name_en,
            'ru': self.first_name_ru,
        }

    def get_inflated_first(self):
        first = self.get_first()
        self.set_name_cases()
        first['ru_cases'] = self._first_name_cases
        return first

    def get_last(self):
        """ Last names for different languages. """
        return {
            'en': self.last_name_en,
            'ru': self.last_name_ru,
        }

    def get_inflated_last(self):
        last = self.get_last()
        self.set_name_cases()
        last['ru_cases'] = self._last_name_cases
        return last

    def get_city(self):
        """ City names for different languages. """
        return {'en': self.city_name_en, 'ru': self.city_name_ru}

    def get_position(self):
        """ Position names for different languages. """
        return {'en': self.official_position_en,
                'ru': self.official_position_ru}

    def get_department_ids(self):
        return self.groups.values_list('staff_id', flat=True)

    def get_email(self):
        return self.login + '@yandex-team.ru'

    def get_primary_department(self):
        for membership in self.groupmembership_set.all():
            if membership.is_primary:
                return membership.group

    def get_controlled_departments(self):
        departments = []
        for membership in self.groupmembership_set.all():
            if membership.is_chief:
                departments.append(membership.group_id)
        for dep in self.hr_departments:
            departments.append(dep.staff_id)
        return departments

    def is_hr(self):
        return bool(self.hr_departments)

    def get_primary_department_hierarchy(self):
        primary = self.get_primary_department()
        if primary is not None:
            return primary.get_ancestors(include_self=True)

    def get_primary_department_hierarchy_ids(self, include_self=True):
        primary = self.get_primary_department()
        if primary is not None:
            return primary.get_ancestors_ids(include_self=include_self)
        return []

    @property
    def heads(self):
        if self._heads is None:
            _heads = get_heads_for_persons([self.staff_id], all_heads=True)
            if _heads is None:
                self._heads = []
            else:
                self._heads = _heads.get(self.staff_id, [])
        if self.staff_id == self.volozh_staff_id:
            self._heads = [self.bunina_staff_id]
        return self._heads

    @property
    def head(self):
        if self._head is None:
            _heads = get_heads_for_persons([self.staff_id], all_heads=False) or {}
            _head = _heads.get(self.staff_id, None)
            if _head:
                self._head = Person.objects.get(staff_id=_head)
        if self.staff_id == self.volozh_staff_id:
            self._head = Person.objects.get(staff_id=self.bunina_staff_id)
        return self._head

    def get_person_subordinates_ids(self):
        from fb.staff.models import Group, GroupMembership
        controlled_departments = self.get_controlled_departments()
        paths = Group.objects.filter(staff_id__in=controlled_departments).values_list('path', flat=True)
        lca_paths = set([])
        for p1 in paths:
            for p2 in paths:
                if p1.startswith(p2) and p1 != p2:
                    break
            else:
                lca_paths.add(p1)

        if lca_paths:
            group_membership_condition = reduce(operator.or_, (
                Q(group_id__in=Group.objects.filter(path__startswith=path))
                for path in lca_paths
            ), Q()) & Q(is_primary=True)
            return GroupMembership.objects.filter(
                group_membership_condition,
            ).exclude(person=self).values_list('person_id', flat=True)
        else:
            return []

    def is_head_of(self, person):
        return any(h.staff_id in person.heads for h in chain([self], self.hr_head_departments))

    def has_read_role(self, person):
        return self.roles.has_read_role(person)

    @property
    def url(self):
        return 'https://staff.yandex-team.ru/{self.login}/'.format(self=self)

    @property
    def avatar_url(self):
        return 'https://center.yandex-team.ru/api/v1/user/{self.login}/avatar/90.jpg'.format(self=self)

    @property
    def goals_url(self):
        return 'https://goals.yandex-team.ru/filter?user={self.staff_id}'.format(self=self)

    @property
    def feedback_url(self):
        return 'https://fb.yandex-team.ru/people?user={self.staff_id}'.format(self=self)

    def __unicode__(self):
        return self.get_full_name()

    def __repr__(self):
        return u"<Person: {0} {1} {2}>".format(self.first_name_en, self.last_name_en, self.pk)

    objects = models.Manager()


class RoleCollection(list):
    def __init__(self, seq=()):
        super(RoleCollection, self).__init__(seq)
        self._can_read = {}

    @staticmethod
    def _check_role(role, person):
        if role.is_person_role:
            if role.subject != person:
                return False
            if role.granter.is_head_of(person):
                return True
        elif role.is_group_role:
            hierarchy = person.get_primary_department_hierarchy_ids()
            if role.group.staff_id in hierarchy:
                return True
        return False

    def has_read_role(self, person):
        if person.staff_id not in self._can_read:
            self._can_read[person.staff_id] = any(
                self._check_role(role, person)
                for role in self
            )

        return self._can_read[person.staff_id]
