from rest_framework import serializers
from rest_framework.fields import Field
from fb.staff.models import Person, GroupDetails, Group


class GroupSerializer(serializers.ModelSerializer):
    name_i18n = Field()
    url = serializers.URLField()

    class Meta:
        model = GroupDetails


class NameSerializer(serializers.Serializer):
    first = Field(source='get_first')
    last = Field(source='get_last')


class InflatedNameSerializer(serializers.Serializer):
    first = Field(source='get_inflated_first')
    last = Field(source='get_inflated_last')


class PersonSerializer(serializers.ModelSerializer):
    name = NameSerializer(source='*', read_only=True)
    city = serializers.Field(source='get_city')
    official_position = serializers.Field(source='get_position')
    official_join_at = serializers.DateField(read_only=True)

    class Meta:
        model = Person
        exclude = (
            'first_name_ru',
            'first_name_en',
            'last_name_ru',
            'last_name_en',
            'groups',
            'city_name_ru',
            'city_name_en',
            'official_position_ru',
            'official_position_en',
        )
        read_only_fields = (
            'uid',
            'login',
            'is_dismissed',
            'gender',
            'staff_id',
            'language_ui',
        )


class SubordinatesPersonSerializer(PersonSerializer):

    head = PersonSerializer()

    def to_native(self, instance):
        ret = super(SubordinatesPersonSerializer, self).to_native(instance)
        ret['departments'] = GroupSerializer(instance.get_primary_department_hierarchy(), many=True).data
        return ret

class InflatedPersonSerializer(PersonSerializer):
    name = InflatedNameSerializer(source='*', read_only=True)

    class Meta(PersonSerializer.Meta):
        pass


class HeadsForPersonSerializer(serializers.Serializer):
    person_id = serializers.IntegerField()
    heads = PersonSerializer(many=True)


class PersonInDepartmentSerializer(PersonSerializer):
    department_groups = GroupSerializer(source='get_primary_department_hierarchy', many=True)
    id = serializers.IntegerField(source='staff_id')
    head = PersonSerializer()

    class Meta:
        model = Person
        fields = (
            'name',
            'login',
            'uid',
            'department_groups',
            'id',
            'is_dismissed',
            'city',
            'official_position',
            'official_join_at',
            'head',
            'language_ui',
        )

    read_only = True


class GroupStructure(serializers.ModelSerializer):
    class Meta:
        model = Group


class StaffHead(serializers.Serializer):
    id = serializers.IntegerField()


class StaffDepartment(serializers.Serializer):
    heads = StaffHead(source='get_heads_dict', many=True)


class ExtendedGroupSerializer(GroupSerializer):
    path = serializers.CharField(source='group_structure.path')
    department = StaffDepartment(source='*')
    heads = PersonSerializer(source='get_heads_list', many=True)
