from itertools import chain
import json
from django.db import transaction, DatabaseError

from celery import shared_task
from django.db.models import Count
from cia_stuff.staff import sync
from cia_stuff.staff import api

from fb.staff.models import Group, Person, GroupDetails, GroupMembership
from fb.common.lock import get_lock_or_do_nothing
from django.conf import settings
from django.contrib.auth import get_user_model

import logging

logger = logging.getLogger(__name__)

SEARCH_SORTING_DICT = dict((value, index) for (index, value) in enumerate(GroupDetails.SEARCH_HEAD_DEPARTMENTS))
YANDEX_SORTING_DICT = dict((value, index) for (index, value) in enumerate(GroupDetails.YANDEX_HEAD_DEPARTMENTS))


def _cmp_search_groups(x, y):
    return cmp(SEARCH_SORTING_DICT.get(x, 999), SEARCH_SORTING_DICT.get(y, 999))


def _cmp_yandex_head_deps(x, y):
    return cmp(YANDEX_SORTING_DICT.get(x, 999), YANDEX_SORTING_DICT.get(y, 999))


def _cmp_groups(x, y):
    return _cmp_yandex_head_deps(x, y) or _cmp_search_groups(x, y)


def _cmp_ancestors_groups(x, y):
    _res = 0
    for (a, b) in zip(x, y):
        _res = _cmp_groups(a, b)
        if _res:
            break
    return _res


def _cmp(x, y):
    _res = cmp(x['count'], y['count'])
    _res = _res or _cmp_ancestors_groups(x['ancestors'], y['ancestors'])
    _res = _res or _cmp_groups(x['id'], y['id'])
    return _res


def _prepare_request_for_load_groups_from_staff():
    url = settings.STAFF_API_BASE_URL + 'groups'
    groups_api = api.ApiWrapper(
        url, settings.STAFF_TOKEN, page_size=5000,
        fields=[
            'department.name.full',
            'department.url',
            'id',
            'group',
            'ancestors.id',
        ],
        filter_args={'type': 'department', '_sort': 'id'},
    )
    return groups_api


def _load_groups(groups):
    for d in groups:
        d['ancestors'] = tuple(x['id'] for x in d['ancestors'])
        d['count'] = len(d['ancestors'])

    node_by_path = {}
    root = {'data': {}, 'children': []}
    for g in sorted(groups, cmp=_cmp):
        node = {
            'data': {
                'staff_id': g['id'],
            },
            'children': [],
        }
        key = g['ancestors'] + (g['id'], )
        node_by_path[key] = node
        try:
            parent = node_by_path[g['ancestors']] if g['ancestors'] else root
        except KeyError:
            logger.critical("Could not add %d! Skipping.", g['id'])
        else:
            parent['children'].append(node)

    with transaction.atomic():
        # FIXME: deletion in sqlite fails.
        for group in groups:
            full_name = group['department']['name']['full']
            GroupDetails(
                staff_id=group['id'],
                name=full_name.get('ru', ''),
                name_en=full_name.get('en', ''),
                machine_name=group['department']['url'],
            ).save()
        try:
            Group.objects.all().delete()
        except DatabaseError:
            for g in Group.objects.all():
                g.delete()
        Group.load_bulk(root['children'])
    return len(groups)


@shared_task
@get_lock_or_do_nothing('load_groups_from_staff')
def load_groups_from_staff():
    groups_api = _prepare_request_for_load_groups_from_staff()
    groups = list(groups_api)
    res = _load_groups(groups)
    load_group_membership_from_staff.delay()
    return res


def _map_fields(iterable_persons):
    for p in iterable_persons:
        new_p = {
            'staff_id': p['id'],
            'uid': p['uid'],
            'login': p['login'],
            'is_dismissed': p['official']['is_dismissed'],
            'is_robot': p['official']['is_robot'],

            'first_name_ru': p['name']['first'].get('ru', ''),
            'last_name_ru': p['name']['last'].get('ru', ''),

            'first_name_en': p['name']['first'].get('en', ''),
            'last_name_en': p['name']['last'].get('en', ''),

            'city_name_ru': p['location']['office']['city']['name'].get('ru', ''),
            'city_name_en': p['location']['office']['city']['name'].get('en', ''),

            'official_position_ru': p['official']['position'].get('ru', ''),
            'official_position_en': p['official']['position'].get('en', ''),
            'official_join_at': p['official'].get('join_at', None),
            'language_ui': p['language'].get('ui') or 'ru',
            'gender': int(p['personal']['gender'] != 'male'),
        }
        yield new_p


def _prepare_request_for_load_persons_from_staff():
    url = settings.STAFF_API_BASE_URL + 'persons'
    persons_api = api.ApiWrapper(
        url, settings.STAFF_TOKEN, page_size=15000,
        filter_args={'_sort': 'id'},
        fields=[
            'id',
            'uid',
            'login',
            'official.is_dismissed',
            'official.is_robot',
            'name.first.ru',
            'name.first.en',
            'name.last.ru',
            'name.last.en',
            'personal.gender',
            'location.office.city.name',
            'official.position', 'official.join_at',
            'language',
        ],
    )
    return persons_api


@shared_task
@get_lock_or_do_nothing('load_persons_from_staff')
def load_persons_from_staff():
    persons_api = _prepare_request_for_load_persons_from_staff()
    persons = [Person(**fields) for fields in _map_fields(persons_api)]

    with transaction.atomic():
        for p in persons:
            _ensure_user(p)
            p.save()
    load_groups_from_staff.delay()
    return len(persons)


def _ensure_user(person):
    User = get_user_model()
    defaults = {'email': person.get_email()}

    kwargs = {
        User.USERNAME_FIELD: person.login,
        'defaults': defaults,
    }

    user, created = User.objects.get_or_create(**kwargs)
    if created:
        user.set_unusable_password()
        user.save()
    return user


def generate_group_memberships(groups, memberships, existing_persons):
    members_by_group = {}
    main_group_by_id = {}
    for membership in memberships:
        group_id = membership['group']['id']
        person_id = membership['person']['id']
        members_by_group.setdefault(group_id, []).append(person_id)
        main_group_by_id[person_id] = group_id

    for g in groups:
        group_id = g['id']
        heads = g['department']['heads']
        chiefs = set(h['person']['id'] for h in heads if h['role'] == 'chief')
        deputies = set(h['person']['id'] for h in heads if h['role'] == 'deputy')
        members = members_by_group.get(group_id, [])
        all_members = set(chain(chiefs, deputies, members))
        for person_id in all_members & existing_persons:
            main_group = main_group_by_id.get(person_id)

            if main_group is None:
                logger.error('No main group for %d', person_id)

            yield {
                'group_id': group_id,
                'person_id': person_id,
                'is_chief': person_id in chiefs,
                'is_deputy': person_id in deputies,
                'is_primary': main_group_by_id.get(person_id) == group_id,
            }


@shared_task
@get_lock_or_do_nothing('load_group_membership_from_staff')
def load_group_membership_from_staff():
    existing_person_ids = set(Person.objects.values_list('staff_id', flat=True))
    groups_api, membership_api = sync.prepare_request_for_load_group_membership_from_staff()

    group_memberships = [
        GroupMembership(**fields)
        for fields in generate_group_memberships(groups_api, membership_api, existing_person_ids)
    ]

    with transaction.atomic():
        try:
            GroupMembership.objects.all().delete()
        except DatabaseError:
            for g in GroupMembership.objects.all():
                g.delete()
        GroupMembership.objects.bulk_create(group_memberships)
        try_to_recover()

    return len(group_memberships)


def _read_data(file_name):
    with open(file_name) as _input:
        data = json.loads(_input.read())
    return data


def try_to_recover():
    has_not_primary = set(GroupMembership.objects.filter(is_primary=False).values_list('person_id', flat=True))
    has_primary = set(GroupMembership.objects.filter(is_primary=True).values_list('person_id', flat=True))
    no_primary = has_not_primary - has_primary

    if no_primary:
        no_pimary_persons = Person.objects.filter(
            staff_id__in=no_primary,
        ).filter(is_dismissed=False).all()

        for person in no_pimary_persons:
            logger.error("No primary for %s", person.login)
        logger.critical("Trying to fix: %d", len(no_primary))

        recoverable_persons = set(no_pimary_persons.annotate(
            c=Count('groupmembership'),
        ).filter(c=1).values_list('staff_id', flat=True))

        fixed = GroupMembership.objects.filter(
            person_id__in=recoverable_persons,
        ).update(is_primary=True)

        not_fixed = no_primary - recoverable_persons
        logger.critical(
            "Updated: %d, not fixed: %d (%s)",
            fixed, len(not_fixed), ", ".join(map(str, not_fixed)),
        )
