from django.db.models import Prefetch, Q
import operator
from rest_framework import generics
from fb.staff.models import GroupDetails, GroupMembership, Group
from fb.staff.serializers import ExtendedGroupSerializer

__all__ = ['DepartmentList', 'DepartmentDetails', 'DepartmentPathDetails', 'GroupSuggest']


class DepartmentList(generics.ListAPIView):

    serializer_class = ExtendedGroupSerializer
    paginate_by = 10
    paginate_by_param = 'page_size'
    max_paginate_by = 5000

    @staticmethod
    def get_ancestor_pathes(request):
        from fb.common.filter_backends import IntegerAttributes
        extractor = IntegerAttributes(attr_name='ancestors')
        ancestors = extractor._get_attr_value(request)
        if not ancestors:
            return []
        return Group.objects.filter(staff_id__in=ancestors).values_list('path', flat=True)

    def get_queryset(self):

        ancestors = self.get_ancestor_pathes(self.request)
        ancestors = (Q(group_structure__path__startswith=p) for p in ancestors)
        ancestors_filter = reduce(operator.ior, ancestors, Q())

        queryset = GroupDetails.objects.filter(
            group_structure__isnull=False,
        ).filter(
            ancestors_filter,
        ).select_related(
            'group_structure',
        ).prefetch_related(
            Prefetch(
                'groupmembership_set',
                queryset=GroupMembership.objects.filter(
                    Q(is_primary=True)
                    | Q(is_chief=True)
                    | Q(is_deputy=True)
                ).order_by(
                    '-is_chief', '-is_deputy',
                ).select_related('person'),
            ),
        ).all()
        return queryset


class DepartmentDetails(generics.RetrieveAPIView):
    queryset = GroupDetails.objects.all()
    serializer_class = ExtendedGroupSerializer
    lookup_field = 'staff_id'


class DepartmentPathDetails(generics.RetrieveAPIView):
    queryset = GroupDetails.objects.all()
    serializer_class = ExtendedGroupSerializer
    lookup_field = 'group_structure__path'


class GroupSuggest(generics.ListAPIView):

    model = GroupDetails
    serializer_class = ExtendedGroupSerializer

    def get_queryset(self):

        queryset = GroupDetails.objects.filter(group_structure__isnull=False)

        if 'q' not in self.request.GET:
            return queryset.none()

        q = self.request.GET['q']

        queryset = queryset.filter(
            Q(name__icontains=q)
            | Q(name_en__icontains=q)
            | Q(machine_name__icontains=q)
        ).distinct()[:10]
        return queryset
