# coding: utf-8
import ast
import operator
from itertools import imap
from django.db.models import Q
from rest_framework import generics
from rest_framework.response import Response
from fb.staff.misc import get_heads_for_persons
from fb.staff.models import Person
from fb.staff.serializers import PersonSerializer, PersonInDepartmentSerializer, HeadsForPersonSerializer
from fb.common.misc import RelationsAutodetect
from fb.common.filter_backends import IntegerAttributes
import logging


__all__ = ['PersonDetails', 'PersonsList', 'IntelligentSuggest', 'HeadsOfPersons', 'HeadsOfPersonsAll', 'PersonSuggest', 'RelationsAutodetectView']

logger = logging.getLogger(__name__)

relations_autodetect = RelationsAutodetect()


class PersonDetails(generics.RetrieveAPIView):
    queryset = Person.objects.all()
    serializer_class = PersonInDepartmentSerializer
    lookup_field = 'staff_id'


class PersonsList(generics.ListAPIView):
    queryset = Person.objects.all()
    paginate_by = 10
    paginate_by_param = 'page_size'
    serializer_class = PersonInDepartmentSerializer
    lookup_field = 'staff_id'
    filter_fields = ('login', 'uid')
    filter_backends = [
        IntegerAttributes(attr_name='staff_id'),
    ]


# TODO: выпилить вьюху либо сделать свою умную логику.
#  Раньше вся логика была на стороне инвайтера, инвайтера больше нет.
#  Оставляю ручку, просто чтобы не ломать фронт
class IntelligentSuggest(generics.ListAPIView):

    model = Person
    serializer_class = PersonSerializer

    def get_queryset(self):
        # Хост inviter.yandex-team.ru уже давно смотрит на easymeeting.
        # По факту, эта ручка уже почти год отвечает пустым списком.
        return Person.objects.none()


class HeadsOfPersons(generics.ListAPIView):
    serializer_class = PersonSerializer

    def get_persons_list(self):
        persons = self.request.QUERY_PARAMS.get('persons', '')
        try:
            persons = ast.literal_eval(persons)
        except StandardError:
            return []
        if not isinstance(persons, tuple):
            persons = [persons]
        return persons

    def get_queryset(self):
        persons = self.get_persons_list()
        heads = get_heads_for_persons(persons)
        head_ids = list(heads.itervalues())
        return Person.objects.filter(staff_id__in=head_ids).all()


class HeadsOfPersonsAll(generics.ListAPIView):
    serializer_class = HeadsForPersonSerializer

    def get_persons_list(self):
        persons = self.request.QUERY_PARAMS.get('persons', '')
        try:
            persons = ast.literal_eval(persons)
        except StandardError:
            return []
        if not isinstance(persons, tuple):
            persons = [persons]
        return persons

    def get_queryset(self):
        persons = self.get_persons_list()
        heads = get_heads_for_persons(persons, all_heads=True)
        head_ids = set()
        for _heads in heads.itervalues():
            if _heads:
                head_ids.update(_heads)

        heads_instances = {
            p.staff_id: p
            for p in Person.objects.filter(staff_id__in=head_ids).all()
        }

        response = []
        for person_id, _heads in heads.iteritems():
            _heads_instances = [
                heads_instances.get(_id, Person())
                for _id in _heads
            ]
            response.append({
                'person_id': person_id,
                'heads': _heads_instances,
            })
        return response


class PersonSuggest(generics.ListAPIView):

    model = Person
    serializer_class = PersonSerializer

    def _q_for_token(self, q):
        return (
            Q(login__icontains=q)
            | Q(first_name_ru__icontains=q)
            | Q(last_name_ru__icontains=q)
            | Q(first_name_en__icontains=q)
            | Q(last_name_en__icontains=q)
            | Q(official_position_ru__icontains=q)
            | Q(official_position_en__icontains=q)
        )

    def get_queryset(self):
        queryset = Person.objects.all()

        if 'q' not in self.request.GET:
            return queryset.none()

        q = self.request.GET['q']
        tokens = q.split()
        condition = reduce(operator.and_, imap(self._q_for_token, tokens), Q())

        queryset = queryset.filter(
            condition
        ).distinct()[:10]
        return queryset


class RelationsAutodetectView(generics.GenericAPIView):

    def get(self, request, *args, **kwargs):
        me = Person.objects.get(uid=self.request.yauser.uid)
        staff_ids = request.GET.get('staff_ids', '').split(',')
        staff_ids = filter(lambda x: x.isdigit(), staff_ids)
        persons = Person.objects.filter(staff_id__in=staff_ids)
        if not persons:
            return Response({})
        relations = relations_autodetect.get_relations(who=me, persons=persons)
        common_relation = set(relation['relation_type'] for relation in relations)
        if len(common_relation) > 1:
            return Response({
                'result': 'partner',
            })
        else:
            return Response({
                'result': common_relation.pop(),
            })
