import logging
import threading
import time

from django.conf import settings
from django.db import models
from django.db.models.signals import post_save, post_delete, m2m_changed
from django.utils import timezone
from django.contrib.postgres.fields import JSONField
from django.contrib.contenttypes.models import ContentType

from model_utils.choices import Choices

from intranet.femida.src.actionlog import serializers
from intranet.femida.src.actionlog.tasks import (
    touch_candidates_task,
    touch_vacancies_task,
)
from intranet.femida.src.attachments.tasks import extract_text_for_attachments
from intranet.femida.src.core.signals import post_update, post_bulk_create
from intranet.femida.src.isearch.tasks import (
    bulk_push_candidates_to_isearch,
    bulk_push_vacancies_to_isearch,
)


logger = logging.getLogger(__name__)


SNAPSHOT_REASONS = Choices(
    (0, 'nothing', 'nothing'),
    (1, 'addition', 'addition'),
    (2, 'change', 'change'),
    (3, 'deletion', 'deletion'),
)


class LogRecord(models.Model):

    action_name = models.CharField(max_length=127, db_index=True)
    action_time = models.DateTimeField(default=timezone.now, db_index=True)
    user = models.ForeignKey(
        to=settings.AUTH_USER_MODEL,
        on_delete=models.PROTECT,
        null=True,
        blank=True,
    )
    context = JSONField(null=True)
    mongo_id = models.CharField(max_length=31)

    def __str__(self):
        return 'LogRecord %d, %s' % (self.id, self.action_name)


class Snapshot(models.Model):

    log_record = models.ForeignKey(
        to=LogRecord,
        on_delete=models.CASCADE,
        related_name='snapshots',
    )
    reason = models.IntegerField(choices=SNAPSHOT_REASONS)
    data = JSONField(null=True)

    obj_type = models.ForeignKey(
        to=ContentType,
        on_delete=models.SET_NULL,
        blank=True,
        null=True,
    )
    obj_id = models.IntegerField()
    # ContentType может и не быть, поэтому запишем хотя бы что-то
    obj_str = models.CharField(max_length=127)

    def __str__(self):
        return 'Snapshot %d of log record %d' % (self.id, self.log_record_id)

    class Meta:
        indexes = [
            models.Index(fields=['obj_id', 'obj_type'], name='actionlog_snapshot_obj_index')
        ]


class ActionLog:

    def __init__(self):
        self._ctx = threading.local()

    def __enter__(self):
        pass

    def __exit__(self, exp_type, exp_value, exp_tr):
        # Если мы получили ValidationError, то эксепшена здесь не будет, так как мы его
        # обработали ранее. При этом писать в лог пустоту тоже смысла нет.
        # https://st.yandex-team.ru/FEMIDA-2370
        if not exp_type and self.ctx.snapshots:
            self.save()
        else:
            self.reset()

    @property
    def ctx(self):
        return self._ctx

    def get_serialized_obj(self, obj):
        return serializers.ObjectSerializer(obj).data

    def is_initialized(self):
        return bool(getattr(self.ctx, 'action_name', None))

    def init(self, action_name, user=None):
        if self.is_initialized():
            msg = 'Lost actionlog. action_name: `%s`'
            logger.error(msg, self.ctx.action_name)

        # Если пользователь не авторизован, нельзя записывать это в actionlog
        if user is not None and not user.is_authenticated:
            user = None

        self.ctx.action_name = action_name
        self.ctx.user = user
        self.ctx.snapshots = {}
        self.ctx.context = {}
        self.model_content_type_map = dict(ContentType.objects.values_list('model', 'id'))
        return self

    def reset(self):
        self.ctx.action_name = None
        self.ctx.user = None
        self.ctx.snapshots = {}
        self.ctx.context = {}

    def add_context(self, data):
        self.ctx.context.update(data)

    def add_object(self, obj, reason):
        obj_id = obj.id
        obj_str = obj.__class__.__name__.lower()
        obj_type_id = self.model_content_type_map.get(obj_str)
        obj_uid = (obj_type_id, obj_str, obj_id)

        snapshot = self.ctx.snapshots.get(obj_uid)
        data = self.get_serialized_obj(obj)

        if not snapshot:
            self.ctx.snapshots[obj_uid] = Snapshot(
                reason=reason,
                data=data,
                obj_type_id=obj_type_id,
                obj_id=obj_id,
                obj_str=obj_str,
            )
        else:
            snapshot.data = data

    def add_related_object(self, obj_id, obj_str):
        obj_type_id = self.model_content_type_map.get(obj_str)
        obj_uid = (obj_type_id, obj_str, obj_id)
        if obj_uid in self.ctx.snapshots:
            return

        self.ctx.snapshots[obj_uid] = Snapshot(
            reason=SNAPSHOT_REASONS.nothing,
            data=None,
            obj_type_id=obj_type_id,
            obj_id=obj_id,
            obj_str=obj_str,
        )

    def save(self):
        start = time.time()

        try:
            log_record = LogRecord.objects.create(
                action_name=self.ctx.action_name,
                user=self.ctx.user,
                context=self.ctx.context,
            )
            for snapshot in self.ctx.snapshots.values():
                snapshot.log_record = log_record
            Snapshot.objects.bulk_create(self.ctx.snapshots.values())
            self.post_save()
        except Exception:
            logger.exception('[ACTION_LOG] Failed to save log record `%s`', self.ctx.action_name)
            raise
        finally:
            self.reset()

        logger.info(
            '[ACTION_LOG] Log record added. Action name: %s, time to save: %s',
            log_record.action_name, time.time() - start
        )

    def post_save(self):
        """
        Воспользоваться тем фактом, что после сейва экшенлога
        у нас есть список объектов, которые как-то связаны с экшеном
        и других манипуляций с базой не будет.
        """
        candidate_ids = set()
        vacancy_ids = set()
        attachment_ids = set()

        nested_candidate_collections = [
            'candidatejob',
            'candidatecontact',
            'candidateeducation',
            'candidateprofession',
            'candidateskill',
            'candidatecity',
            'candidatetag',
            'application',
            'interview',
            'consideration',
            'candidateresponsible',
        ]

        for snapshot in self.ctx.snapshots.values():
            data = snapshot.data
            if data is None:
                continue
            if snapshot.obj_str == 'candidate':
                candidate_ids.add(data['id'])
            elif snapshot.obj_str in nested_candidate_collections:
                candidate_ids.add(data['candidate'])
            elif snapshot.obj_str == 'vacancy':
                vacancy_ids.add(data['id'])
            elif snapshot.obj_str == 'vacancymembership':
                vacancy_ids.add(data['vacancy'])
            elif snapshot.obj_str == 'candidateattachment':
                attachment_ids.add(data['attachment'])

        candidate_ids = list(candidate_ids)
        vacancy_ids = list(vacancy_ids)
        attachment_ids = list(attachment_ids)

        if candidate_ids:
            touch_candidates_task.delay(candidate_ids)
            bulk_push_candidates_to_isearch.delay(candidate_ids)
        if vacancy_ids:
            touch_vacancies_task.delay(vacancy_ids)
            bulk_push_vacancies_to_isearch.delay(vacancy_ids)
        if attachment_ids:
            extract_text_for_attachments.delay(attachment_ids)


actionlog = ActionLog()


from intranet.femida.src.actionlog import handlers  # noqa: E402


post_save.connect(handlers.actionlog_callback)
post_delete.connect(handlers.actionlog_callback)
post_update.connect(handlers.actionlog_update_callback)
post_bulk_create.connect(handlers.actionlog_bulk_create_callback)
m2m_changed.connect(handlers.actionlog_m2m_changed_callback)
