from django.core.exceptions import ValidationError
import sform

from intranet.femida.src.api.core.forms import (
    CandidateSuggestField,
    VacancyMultipleSuggestField,
    SortField,
    EMPTY_LABEL,
    AttachmentMultipleSuggestField,
    InterviewMultipleChoiceField,
    CounterModelChoiceField,
    CounterChoiceField,
    ConsiderationSuggestField,
)
from intranet.femida.src.applications.helpers import (
    count_applications_per_hiring_stage,
    count_applications_per_vacancy,
)
from intranet.femida.src.interviews import choices
from intranet.femida.src.interviews.models import Interview
from intranet.femida.src.offers.choices import SOURCES
from intranet.femida.src.vacancies.models import Vacancy
from intranet.femida.src.candidates.choices import CANDIDATE_STATUSES


class CommentForm(sform.SForm):
    """
    Для создания сообщения в коммуникации
    """
    comment = sform.CharField()


class ApplicationResolutionChoiceField(sform.ChoiceField):

    def structure_as_dict(self, *args, **kwargs):
        structure = super().structure_as_dict(*args, **kwargs)
        structure['type'] = 'choice'
        for choice in structure['choices']:
            if choice['value'] in choices.IN_PROGRESS_APPLICATION_RESOLUTIONS:
                choice['status'] = choices.APPLICATION_STATUSES.in_progress
            elif choice['value'] in choices.CLOSED_APPLICATION_RESOLUTIONS_WITH_DEPRECATED:
                choice['status'] = choices.APPLICATION_STATUSES.closed
        return structure


class ApplicationBulkCreateForm(sform.SForm):

    candidate = CandidateSuggestField(state=sform.REQUIRED)
    rank = sform.IntegerField(max_value=1000, default=0)
    comment = sform.CharField()
    create_activated = sform.BooleanField()
    source = sform.ChoiceField(
        choices=SOURCES,
        state=sform.REQUIRED,
        empty_label=EMPTY_LABEL,
    )
    source_description = sform.CharField(max_length=255)
    interviews = InterviewMultipleChoiceField(queryset=Interview.unsafe.none())
    vacancies = VacancyMultipleSuggestField(state=sform.REQUIRED)

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        assert 'user' in self.context
        self.user = self.context['user']
        if self.user.is_recruiter:
            interviews = self.context.get('interviews')
            assert interviews is not None
            self.fields['interviews'].queryset = interviews
        else:
            self.fields.pop('interviews', None)
            self.fields.pop('create_activated', None)
            self.fields['source'].default = SOURCES.found_by_hiring_manager

        candidate = self.context.get('candidate')
        if candidate is not None:
            self.fields['candidate'].default = candidate.id
            if candidate.status == CANDIDATE_STATUSES.in_progress:
                self.fields.pop('source', None)
                self.fields.pop('source_description', None)


class ApplicationActivateForm(CommentForm, sform.SForm):

    interviews = InterviewMultipleChoiceField(queryset=Interview.unsafe.none())

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        interviews = self.context.get('interviews')
        assert interviews is not None
        self.fields['interviews'].queryset = interviews


class ApplicationCloseForm(CommentForm, sform.SForm):

    resolution = sform.ChoiceField(
        choices=choices.CLOSED_APPLICATION_RESOLUTIONS_MANAGEABLE,
        state=sform.REQUIRED,
        empty_label=EMPTY_LABEL,
    )


class ApplicationChangeResolutionForm(CommentForm, sform.SForm):

    resolution = sform.ChoiceField(
        choices=choices.CLOSED_APPLICATION_RESOLUTIONS_MANAGEABLE,
        state=sform.REQUIRED,
        empty_label=EMPTY_LABEL,
    )


class ApplicationListFilterBaseForm(sform.SForm):
    """
    Базовая форма, которая используется в ручке формы фильтра

    TODO: Со статусами получилось плохо, потому что в ручке формы мы отдаем дефолтное значение
    для статуса, а по факту при фильтрации это дефолтное значение учитывать не будем
    """
    status = sform.ChoiceField(
        choices=choices.APPLICATION_FILTER_FORM_STATUSES,
        default=choices.APPLICATION_FILTER_FORM_STATUSES.open,
        empty_label=EMPTY_LABEL,
    )
    resolution = ApplicationResolutionChoiceField(
        choices=choices.APPLICATION_RESOLUTIONS,
        empty_label=EMPTY_LABEL,
    )
    source = sform.ChoiceField(
        choices=choices.APPLICATION_SOURCES,
        empty_label=EMPTY_LABEL,
    )
    proposal_status = sform.ChoiceField(
        choices=choices.APPLICATION_PROPOSAL_STATUSES,
        empty_label=EMPTY_LABEL,
    )
    is_created_from_submission = sform.NullBooleanField()  # DEPRECATED
    candidate_has_interviews = sform.NullBooleanField()
    sort = SortField(
        choices=choices.APPLICATION_SORTING_TYPES,
        default='-modified',
    )


class ApplicationListFilterForm(ApplicationListFilterBaseForm):
    """
    Форма, по которой фактически происходит фильтрация списка прет-в
    """
    status = sform.ChoiceField(
        choices=choices.APPLICATION_FILTER_FORM_STATUSES,
        empty_label=EMPTY_LABEL,
    )
    consideration = ConsiderationSuggestField()
    vacancy_id = sform.IntegerField()

    # TODO: выпилить после релиза FEMIDA-4778
    consideration_id = ConsiderationSuggestField()

    def clean(self):
        cleaned_data = super().clean()
        # Лучше даже не пытаться фильтровать прет-ва
        # без рассмотрения либо вакансии - запросы плохие.
        required_fields = (
            'consideration',
            'consideration_id',
            'vacancy_id',
        )
        if not any(cleaned_data.get(f) for f in required_fields):
            raise ValidationError(
                message='Specify one of required fields: {}'.format(', '.join(required_fields)),
                code='missing_consideration_vacancy',
            )

        if cleaned_data.get('consideration_id'):
            cleaned_data['consideration'] = cleaned_data.pop('consideration_id')

        return cleaned_data


class InternalMessageCreateForm(sform.SForm):

    text = sform.CharField(state=sform.REQUIRED)
    attachments = AttachmentMultipleSuggestField()


class ApplicationHiringListFilterForm(sform.SForm):

    vacancy = CounterModelChoiceField(
        queryset=Vacancy.unsafe.none(),
        label_extractor=lambda x: x.name,
        empty_label=EMPTY_LABEL,
    )
    stage = CounterChoiceField(
        choices=choices.APPLICATION_HIRING_STAGES,
        empty_label=EMPTY_LABEL,
    )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        applications = self.context['applications']

        # Берём только вакансии, на которых есть претенденты
        self.fields['vacancy'].queryset = (
            Vacancy.unsafe
            .filter(id__in=applications.values('vacancy_id'))
            .order_by('name')
        )

    def structure_as_dict(self, *args, **kwargs):
        applications = stage_applications = self.context['applications']

        self.fields['vacancy'].counts = count_applications_per_vacancy(applications)

        # Берём только прет-ва на выбранной вакансии
        if self.initial.get('vacancy'):
            stage_applications = stage_applications.filter(vacancy=self.initial['vacancy'])
        self.fields['stage'].counts = count_applications_per_hiring_stage(stage_applications)

        result = super().structure_as_dict(*args, **kwargs)
        return result
