from django.db.models import Prefetch
from django.conf import settings
from rest_framework import serializers

from intranet.femida.src.api.candidates.serializers import (
    CandidateLiteSerializer,
    ConsiderationLiteSerializer,
    CandidateBaseSerializer,
)
from intranet.femida.src.api.candidates.utils import get_unique_serialized_attachments
from intranet.femida.src.api.core.serializers import (
    FemidaSerializer,
    WorkflowActionsField,
    IdNameSerializer,
    AwareSerializerMixin,
)
from intranet.femida.src.api.professions.serializers import ProfessionLiteSerializer
from intranet.femida.src.api.staff.serializers import DepartmentSerializer
from intranet.femida.src.api.submissions.serializers import SubmissionLiteSerializer
from intranet.femida.src.api.users.serializers import UserSerializer
from intranet.femida.src.api.vacancies.serializers import VacancyLiteSerializer
from intranet.femida.src.applications.models import Application
from intranet.femida.src.candidates.choices import CONSIDERATION_STATUSES
from intranet.femida.src.candidates.considerations.helpers import (
    add_items_counts_to_considerations_qs,
)
from intranet.femida.src.candidates.helpers import get_candidate_professions_prefetch
from intranet.femida.src.candidates.models import Consideration, Candidate, CandidateSubmission
from intranet.femida.src.interviews import choices
from intranet.femida.src.applications.workflow import ApplicationWorkflow
from intranet.femida.src.vacancies.choices import VACANCY_TYPES
from intranet.femida.src.vacancies.models import Vacancy


def _get_last_comment(obj):
    comment = obj.last_comment
    if comment is None:
        return None
    comment['author'] = comment.pop('author_')
    return comment


def _get_interviews_counts(obj):
    if obj.interviews_counts is None:
        return {}
    return {
        elem['type']: elem['count']
        for elem in obj.interviews_counts
    }


class VacancyForApplicationSerializer(FemidaSerializer):
    """
    Вспомогательный сериализатор. Требует дополнительного указания select/perfect_related.
    """
    department = DepartmentSerializer()
    cities = IdNameSerializer(many=True)
    is_internship = serializers.SerializerMethodField()
    profession = ProfessionLiteSerializer()
    skills = IdNameSerializer(many=True)
    is_available = serializers.SerializerMethodField()

    def get_is_internship(self, obj):
        return obj.type == VACANCY_TYPES.internship

    def get_is_available(self, obj):
        return self.root.context['is_vacancy_available']

    class Meta:
        model = Vacancy
        fields = (
            'id',
            'name',
            'department',
            'status',
            'cities',
            'is_internship',
            'profession',
            'skills',
            'is_available',
        )


class CandidateForApplicationSerializer(CandidateBaseSerializer):
    """
    Вспомогательный сериализатор. Требует дополнительного указания select/perfect_related.
    """
    resume = serializers.SerializerMethodField()

    def get_resume(self, obj):
        attachments = get_unique_serialized_attachments(obj)
        return attachments[0] if attachments else None

    class Meta:
        model = Candidate
        fields = (
            'id',
            'first_name',
            'last_name',
            'login',
            'city',
            'target_cities',
            'responsibles',
            'jobs',
            'skills',
            'candidate_professions',
            'resume',
        )


class ApplicationSerializer(AwareSerializerMixin, FemidaSerializer):

    candidate = CandidateLiteSerializer()
    consideration = ConsiderationLiteSerializer()
    vacancy = VacancyLiteSerializer()
    created_by = UserSerializer()
    submission = SubmissionLiteSerializer()
    last_comment = serializers.SerializerMethodField()
    comments_count = serializers.IntegerField()
    interviews_counts = serializers.SerializerMethodField()
    actions = WorkflowActionsField(workflow_class=ApplicationWorkflow)

    def get_last_comment(self, obj):
        return _get_last_comment(obj)

    def get_interviews_counts(self, obj):
        return _get_interviews_counts(obj)

    class Meta:
        model = Application
        fields = (
            'id',
            'modified',
            'candidate',
            'consideration',
            'created',
            'created_by',
            'vacancy',
            'status',
            'proposal_status',
            'resolution',
            'submission',
            'rank',
            'last_comment',
            'comments_count',
            'interviews_counts',
            'is_archived',
            'actions',
            'source',
        )

        select_related_map = {
            'candidate': ('candidate',),
            'vacancy': ('vacancy__department',),
            'created_by': ('created_by',),
        }

        prefetch_related_map = {
            'actions': (
                'vacancy__memberships__member',
            ),
            'submission': (
                Prefetch(
                    lookup='submission',
                    queryset=CandidateSubmission.unsafe.select_related(
                        'form',
                        'reference',
                        'rotation',
                        'publication',
                    ),
                ),
            ),
            'consideration': (
                Prefetch(
                    lookup='consideration',
                    queryset=add_items_counts_to_considerations_qs(
                        qs=Consideration.unsafe.all(),
                    ),
                ),
                'consideration__responsibles',
            ),
        }


class ApplicationDetailSerializer(AwareSerializerMixin, FemidaSerializer):

    submission = SubmissionLiteSerializer(context={'fields': ('id', 'source')})
    candidate = CandidateForApplicationSerializer()
    vacancy = VacancyForApplicationSerializer()
    consideration = ConsiderationLiteSerializer(
        context={
            'fields': ('id', 'status', 'extended_status'),
        }
    )
    actions = WorkflowActionsField(workflow_class=ApplicationWorkflow)
    created_by = UserSerializer()

    class Meta:
        model = Application
        fields = (
            'id',
            'status',
            'proposal_status',
            'resolution',
            'source',
            'created',
            'modified',
            'submission',

            'candidate',
            'vacancy',
            'consideration',
            'actions',
            'created_by',
        )

        select_related_map = {
            'candidate': ('candidate',),
            'vacancy': ('vacancy',),
            'consideration': ('consideration',),
            'created_by': ('created_by',),
            'submission': ('submission',),
        }

        prefetch_related_map = {
            'candidate': (
                'candidate__target_cities',
                'candidate__skills',
                'candidate__attachments',
                'candidate__jobs',
                'candidate__responsibles',
                get_candidate_professions_prefetch('candidate'),
            ),
            'vacancy': (
                'vacancy__profession__professional_sphere',
                'vacancy__department',
                'vacancy__skills',
                'vacancy__cities',
            ),
        }


class CandidateApplicationSerializer(FemidaSerializer):

    vacancy = VacancyLiteSerializer()
    created_by = UserSerializer()
    submission = SubmissionLiteSerializer()
    last_comment = serializers.SerializerMethodField()
    comments_count = serializers.IntegerField()
    interviews_counts = serializers.SerializerMethodField()
    actions = WorkflowActionsField(workflow_class=ApplicationWorkflow)

    def get_last_comment(self, obj):
        return _get_last_comment(obj)

    def get_interviews_counts(self, obj):
        return _get_interviews_counts(obj)

    class Meta:
        model = Application
        fields = (
            'id',
            'modified',
            'created',
            'created_by',
            'vacancy',
            'status',
            'proposal_status',
            'resolution',
            'submission',
            'actions',
            'source',
            'last_comment',
            'comments_count',
            'interviews_counts',
        )


class ApplicationBulkSerializer(serializers.Serializer):

    items = serializers.SerializerMethodField()
    warnings = serializers.ListField()

    def get_items(self, obj):
        return ApplicationSerializer(
            instance=obj['items'],
            many=True,
            context=self.root.context,
        ).data


class ApplicationAsChoiceSerializer(serializers.Serializer):

    label = serializers.SerializerMethodField()
    vacancy = serializers.SerializerMethodField()
    is_consideration_archived = serializers.SerializerMethodField()

    def get_label(self, obj):
        return str(obj)

    def get_vacancy(self, obj):
        return {
            'id': obj.vacancy.id,
            'name': obj.vacancy.name,
            'startrek_key': obj.vacancy.startrek_key,
            'is_code': obj.vacancy.professional_sphere_id == settings.DEVELOPMENT_PROF_SPHERE_ID,
        }

    def get_is_consideration_archived(self, obj):
        return obj.consideration.state == CONSIDERATION_STATUSES.archived


# FORM SERIALIZERS


class ApplicationListFilterFormSerializer(serializers.Serializer):

    status = serializers.ChoiceField(choices=choices.APPLICATION_FILTER_FORM_STATUSES)
    resolution = serializers.ChoiceField(choices=choices.APPLICATION_RESOLUTIONS)
    is_created_from_submission = serializers.NullBooleanField()
    candidate_has_interviews = serializers.NullBooleanField()
    sort = serializers.ChoiceField(choices=choices.APPLICATION_SORTING_TYPES)
    source = serializers.ChoiceField(choices=choices.APPLICATION_SOURCES)
    proposal_status = serializers.ChoiceField(choices=choices.APPLICATION_PROPOSAL_STATUSES)


class ApplicationHiringListFilterFormSerializer(serializers.Serializer):

    vacancy = serializers.IntegerField(source='vacancy.id', default=None)
    stage = serializers.ChoiceField(choices=choices.APPLICATION_HIRING_STAGES)
