from django.core.exceptions import PermissionDenied
from django.utils.functional import cached_property

from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.api.applications import forms, serializers
from intranet.femida.src.api.candidates.permissions import NoDuplicatesPermission
from intranet.femida.src.api.core.views import (
    BaseView,
    BaseFormViewMixin,
    QueryParamsFormView,
    QueryParamsFormViewMixin,
)
from intranet.femida.src.api.core.views import WorkflowView
from intranet.femida.src.api.hire_orders.permissions import ForbiddenForAutohire
from intranet.femida.src.api.offers.serializers import OfferSerializer
from intranet.femida.src.applications import models
from intranet.femida.src.applications.helpers import (
    get_candidate_visible_interviews,
    annotate_application_qs_for_serialization,
    filter_applications_by_stage,
)
from intranet.femida.src.api.candidates.mixins import CandidateHelperFormMixin
from intranet.femida.src.applications.workflow import ApplicationWorkflow
from intranet.femida.src.candidates.controllers import get_candidates_extended_statuses
from intranet.femida.src.candidates.helpers import candidate_is_current_employee_subquery
from intranet.femida.src.candidates.considerations.helpers import get_considerations_items_counts
from intranet.femida.src.candidates.models import Candidate
from intranet.femida.src.core.shortcuts import get_object_or_40x
from intranet.femida.src.vacancies.choices import VACANCY_ROLES
from intranet.femida.src.vacancies.models import Vacancy, VacancyMembership
from intranet.femida.src.interviews.choices import (
    APPLICATION_FILTER_FORM_STATUSES,
    APPLICATION_STATUSES,
    OPEN_APPLICATION_STATUSES,
)
from intranet.femida.src.interviews.models import Application, Interview

from intranet.femida.src.communications.models import Message
from intranet.femida.src.communications.choices import COMMENT_MESSAGE_TYPES
from intranet.femida.src.communications.helpers import prefetch_user_reminders
from intranet.femida.src.api.communications import serializers as message_serializers
from intranet.femida.src.communications.controllers import update_or_create_internal_message


class ApplicationListView(BaseView):

    model_class = models.Application
    list_item_serializer_class = serializers.ApplicationSerializer

    def filter_queryset(self, queryset):
        filter_form = forms.ApplicationListFilterForm(data=self.request.query_params)
        self.validate(filter_form)
        filter_params = filter_form.cleaned_data

        # Если мы получили рассмотрение, то проверять доступы смысла уже нет
        # - они одинаковые и для прет-в, и для рассмотрений
        if filter_params['consideration']:
            queryset = Application.unsafe.filter(consideration=filter_params['consideration'])

        queryset = annotate_application_qs_for_serialization(queryset)

        simple_filter_fields = (
            'vacancy_id',
            'resolution',
            'source',
            'proposal_status',
        )
        queryset = queryset.filter(**{
            k: v for k, v in filter_params.items() if k in simple_filter_fields and v
        })

        if filter_params['status']:
            if filter_params['status'] == APPLICATION_FILTER_FORM_STATUSES.open:
                queryset = queryset.filter(status__in=OPEN_APPLICATION_STATUSES._db_values)
            else:
                queryset = queryset.filter(status=filter_params['status'])

        if filter_params['is_created_from_submission'] is not None:
            queryset = queryset.filter(
                submission__isnull=not filter_params['is_created_from_submission'],
            )

        # TODO: флаг желательно переименовать, т.к. теперь он несет
        # в себе не совсем то значение, что было изначально.
        # Теперь фильтруем не по секциям кандидата, а по секциям прет-ва.
        # Можно назвать просто как-то вроде interviews_exist
        if filter_params['candidate_has_interviews'] is not None:
            applicatons_with_interviews_qs = (
                Interview.unsafe.alive()
                .filter(application__isnull=False)
                .values('application')
            )
            if filter_params['candidate_has_interviews']:
                queryset = queryset.filter(id__in=applicatons_with_interviews_qs)
            else:
                queryset = queryset.exclude(id__in=applicatons_with_interviews_qs)

        if filter_params['sort']:
            queryset = queryset.order_by(filter_params['sort'], 'id')
        return queryset

    def get(self, request, *args, **kwargs):
        """
        Список претенденств
        """
        return self.list(request, *args, **kwargs)

    def get_list_item_serializer_context(self):
        vacancy_ids = [i.vacancy_id for i in self.page]
        candidate_ids = [i.candidate_id for i in self.page]
        available_vacancy_ids = set(
            Vacancy.objects.filter(id__in=vacancy_ids).values_list('id', flat=True)
        )
        return {
            'available_vacancy_ids': available_vacancy_ids,
            'extended_statuses': get_candidates_extended_statuses(candidate_ids),
            'current_employee_ids': set(
                Candidate.unsafe
                .annotate(is_current_employee=candidate_is_current_employee_subquery)
                .filter(
                    id__in=candidate_ids,
                    is_current_employee=True,
                )
                .values_list('id', flat=True)
            )
        }


class ApplicationDetailView(BaseView):

    model_class = Application
    detail_serializer_class = serializers.ApplicationDetailSerializer

    def get(self, request, *args, **kwargs):
        return self.retrieve(request, *args, **kwargs)

    def get_detail_serializer_context(self):
        return {
            'is_vacancy_available': (
                Vacancy.objects
                .filter(
                    id=self.get_object().vacancy.id,
                )
                .exists()
            ),
        }


class ApplicationMessageListCreateView(BaseFormViewMixin, BaseView):
    """
    Внутренняя коммуникация (комментарии на претенденте)
    """
    model_class = Message
    list_item_serializer_class = message_serializers.InternalMessageSerializer
    detail_serializer_class = message_serializers.InternalMessageSerializer
    validator_class = forms.InternalMessageCreateForm

    @cached_property
    def application(self):
        return get_object_or_40x(Application, pk=self.kwargs.get('application_id'))

    def get_queryset(self):
        # Т.к. при получении прет-ва доступы проверяются,
        # для списка сообщений их проверять смысла нет
        return (
            Message.unsafe
            .alive()
            .filter(application=self.application)
        )

    def filter_queryset(self, queryset):
        queryset = (
            queryset
            .filter(
                type__in=COMMENT_MESSAGE_TYPES._db_values,
            )
            .select_related(
                'author',
                'candidate',
                'application__vacancy__department',
            )
            .prefetch_related(
                'message_attachments__attachment',
                prefetch_user_reminders(self.request.user),
            )
            .order_by('created')
        )
        return queryset

    def get(self, request, *args, **kwargs):
        return self.list(request, *args, **kwargs)

    @action_logged('application_message_create')
    def post(self, request, *args, **kwargs):
        return self.create(request, *args, **kwargs)

    def perform_create(self, data):
        data['application'] = self.application
        data['candidate'] = self.application.candidate
        return update_or_create_internal_message(data=data, initiator=self.request.user)


class ApplicationBulkCreateView(CandidateHelperFormMixin, BaseFormViewMixin, BaseView):

    model_class = models.Application
    validator_class = forms.ApplicationBulkCreateForm
    detail_serializer_class = serializers.ApplicationBulkSerializer
    permission_classes = [
        NoDuplicatesPermission,
        ForbiddenForAutohire('application_bulk_create'),
    ]

    def get_detail_serializer_context(self):
        consideration_ids = self.candidate.considerations.values('id')
        return {
            'considerations_items_counts': get_considerations_items_counts(consideration_ids),
        }

    def get_validator_context(self):
        return {
            'candidate': self.candidate,
            'interviews': get_candidate_visible_interviews(self.candidate),
        }

    def get_detail_serializer_object(self, instance, **kwargs):
        return super().get_detail_serializer_object(
            instance={
                'items': instance,
                'warnings': self.action.extra_data.get('warnings', []),
            },
            **kwargs
        )

    @action_logged('application_bulk_create')
    def post(self, request, *args, **kwargs):
        """
        Создание претенденств на группу вакансий
        """
        return self.create(request, *args, **kwargs)

    def perform_create(self, data):
        validation_vacancy = data['vacancies'][0] if len(data['vacancies']) == 1 else None
        workflow = ApplicationWorkflow(
            instance=None,
            user=self.request.user,
            candidate=data.get('candidate'),
            vacancy=validation_vacancy,
        )

        self.action = workflow.get_action('bulk_create')
        if not self.action.is_available():
            raise PermissionDenied('workflow_error')

        return self.action.perform(**data)


class ApplicationFilterFormView(QueryParamsFormView):

    model_class = Application
    validator_class = forms.ApplicationListFilterBaseForm
    form_serializer_class = serializers.ApplicationListFilterFormSerializer


class ApplicationHiringListView(QueryParamsFormViewMixin, BaseView):

    model_class = Application
    validator_class = forms.ApplicationHiringListFilterForm
    list_item_serializer_class = serializers.ApplicationSerializer
    form_serializer_class = serializers.ApplicationHiringListFilterFormSerializer

    @cached_property
    def vacancy_ids(self):
        return (
            VacancyMembership.unsafe
            .filter(
                member=self.request.user,
                role__in=(
                    VACANCY_ROLES.hiring_manager,
                    VACANCY_ROLES.head,
                    VACANCY_ROLES.responsible,
                    VACANCY_ROLES.main_recruiter,
                    VACANCY_ROLES.recruiter,
                ),
            )
            .values_list('vacancy_id', flat=True)
        )

    def get_queryset(self):
        # Немного упрощаем запрос.
        # Т.к. мы фильтруем по vacancy_ids,
        # мы уже знаем, что у пользователя есть доступ к этим прет-вам,
        # за исключением случая, когда пользователь сам претендует
        # на вакансию, в которой он один из ответственных.
        # Звучит парадоксально, но учтём это условие здесь.
        return (
            self.model_class.unsafe
            .exclude(candidate__login=self.request.user.username)
            .filter(
                status=APPLICATION_STATUSES.in_progress,
                vacancy__in=self.vacancy_ids,
            )
        )

    def filter_queryset(self, queryset):
        queryset = annotate_application_qs_for_serialization(queryset)
        filter_form = self.get_validator_object(self.request.query_params)
        self.validate(filter_form)
        filter_data = filter_form.cleaned_data

        if filter_data['vacancy']:
            queryset = queryset.filter(vacancy=filter_data['vacancy'])

        if filter_data['stage']:
            queryset = filter_applications_by_stage(queryset, filter_data['stage'])

        return queryset.order_by('-modified')

    def get_validator_context(self):
        return {
            'applications': self.get_queryset(),
        }

    def get(self, request, *args, **kwargs):
        return self.list(request, *args, **kwargs)


class ApplicationWorkflowView(WorkflowView):

    model_class = models.Application
    detail_serializer_class = serializers.ApplicationSerializer
    workflow_class = ApplicationWorkflow

    @property
    def actionlog_name(self):
        return 'application_' + self.action_name


class ApplicationActivateView(BaseFormViewMixin, ApplicationWorkflowView):

    action_name = 'activate'
    validator_class = forms.ApplicationActivateForm

    def get_validator_context(self):
        return {
            'interviews': get_candidate_visible_interviews(self.get_object().candidate),
        }


class ApplicationCloseView(BaseFormViewMixin, ApplicationWorkflowView):

    action_name = 'close'
    validator_class = forms.ApplicationCloseForm
    permission_classes = [ForbiddenForAutohire]


class ApplicationReopenView(BaseFormViewMixin, ApplicationWorkflowView):

    action_name = 'reopen'
    validator_class = forms.ApplicationActivateForm

    def get_validator_context(self):
        return {
            'interviews': get_candidate_visible_interviews(self.get_object().candidate),
        }


class ApplicationChangeResolutionView(BaseFormViewMixin, ApplicationWorkflowView):

    action_name = 'change_resolution'
    validator_class = forms.ApplicationChangeResolutionForm


class CreateOfferView(ApplicationWorkflowView):
    """
    Создание оффера на основе претендентства
    """
    detail_serializer_class = OfferSerializer
    action_name = 'create_offer'


class ApplicationProposalView(BaseFormViewMixin, ApplicationWorkflowView):

    validator_class = forms.CommentForm
