from django.contrib.auth import get_user_model

from rest_framework import permissions

from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.api.core.views import BaseView, InstanceFormViewMixin, BaseFormViewMixin
from intranet.femida.src.api.problems.permissions import ProblemPermission
from intranet.femida.src.core.controllers import update_instance
from intranet.femida.src.core.shortcuts import get_object_or_40x
from intranet.femida.src.core.workflow import WorkflowError
from intranet.femida.src.interviews.models import Interview, Assignment
from intranet.femida.src.problems.helpers import get_active_complaints_counts, get_problems_stats
from intranet.femida.src.problems.utils import get_problems_usage_in_context

from .permissions import AssignmentPermission
from . import serializers, forms


User = get_user_model()


class AssignmentListCreateView(BaseFormViewMixin, BaseView):

    model_class = Assignment
    list_item_serializer_class = serializers.AssignmentSerializer
    detail_serializer_class = serializers.AssignmentSerializer
    validator_class = forms.AssignmentCreateForm
    permission_classes = [permissions.IsAuthenticated, AssignmentPermission]

    def get(self, *args, **kwargs):
        """
        Список привязанных к собеседованию задач
        """
        return self.list(*args, **kwargs)

    @action_logged('assignment_create')
    def post(self, *args, **kwargs):
        """
        Привязывание задачи к собеседованию
        """
        return self.create(*args, **kwargs)

    @property
    def interview(self):
        if hasattr(self, '_interview'):
            return self._interview
        self._interview = get_object_or_40x(Interview, pk=self.kwargs.get('interview_id'))
        return self._interview

    def _get_serializer_context(self):
        problems_usage_in_context = get_problems_usage_in_context(interview=self.interview)
        problem_ids = [a.problem_id for a in self.interview.assignments.all()]
        return {
            'problems_usage_in_context': problems_usage_in_context,
            'active_complaints_counts': get_active_complaints_counts(problem_ids),
            'problems_stats': get_problems_stats(problem_ids),
        }

    def get_list_item_serializer_context(self):
        return self._get_serializer_context()

    def get_detail_serializer_context(self):
        return self._get_serializer_context()

    def get_queryset(self):
        queryset = (
            self.interview.assignments
            .select_related('problem')
            .prefetch_related('problem__categories')
            .prefetch_related('problem__created_by')
            .prefetch_related('problem__fans')
        )

        user = self.request.user
        for assignment in queryset:
            problem = assignment.problem
            problem.is_favorite = user in problem.fans.all()

        return queryset

    def perform_create(self, data):
        data['interview'] = self.interview
        return self.model_class.objects.create(**data)


class AssignmentDetailView(InstanceFormViewMixin, BaseView):

    model_class = Assignment
    validator_class = forms.AssignmentForm
    detail_serializer_class = serializers.AssignmentSerializer
    form_serializer_class = serializers.AssignmentFormSerializer
    permission_classes = [
        permissions.IsAuthenticated,
        AssignmentPermission,
        ProblemPermission,
    ]

    def get_detail_serializer_context(self):
        instance = self.get_object()
        problems_usage_in_context = get_problems_usage_in_context(interview=self.interview)
        problem_ids = [instance.problem_id]
        return {
            'problems_usage_in_context': problems_usage_in_context,
            'active_complaints_counts': get_active_complaints_counts(problem_ids),
            'problems_stats': get_problems_stats(problem_ids),
        }

    def get_validator_context(self):
        context = super().get_validator_context()
        context['interview'] = self.interview
        return context

    @action_logged('assignment_update')
    def put(self, request, *args, **kwargs):
        """
        Редактирование привязки задачи к собеседованию
        """
        return self.update(request, *args, **kwargs)

    @action_logged('assignment_update')
    def patch(self, request, *args, **kwargs):
        """
        Частичное редактирование привязки задачи к собеседованию
        """
        return self.partial_update(request, *args, **kwargs)

    @action_logged('assignment_delete')
    def delete(self, request, *args, **kwargs):
        """
        Отвязывание задачи от собеседования
        """
        return self.destroy(request, *args, **kwargs)

    @property
    def interview(self):
        if not hasattr(self, '_interview'):
            self._interview = self.get_object().interview
        return self._interview

    def perform_update(self, data, instance):
        return update_instance(instance, data)

    def perform_destroy(self, instance):
        if instance.interview.state == Interview.STATES.finished:
            raise WorkflowError('interview_finished')
        instance.delete()
