import logging
import mimetypes
import os

from urllib.parse import quote

from django.http import HttpResponse
from django.contrib.auth import get_user_model
from rest_framework import status
from rest_framework.response import Response
from rest_framework.parsers import MultiPartParser, FormParser
from rest_framework.exceptions import PermissionDenied

from intranet.femida.src.api.core import errors
from intranet.femida.src.api.core.permissions import ServicePermission
from intranet.femida.src.api.core.views import BaseView
from intranet.femida.src.core.shortcuts import get_object_or_40x
from intranet.femida.src.permissions.context import context
from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.attachments.models import Attachment
from intranet.femida.src.attachments.helpers import is_attachment_available

if os.getenv('IS_ARCADIA'):
    from django_mds.client import APIError
else:
    from mds import APIError


logger = logging.getLogger(__name__)


User = get_user_model()


class FileUploadView(BaseView):

    parser_classes = (MultiPartParser, FormParser,)

    @action_logged('file_upload')
    def put(self, request, *args, **kwargs):
        """
        Заливка файла
        """
        try:
            file = request.FILES['file']
            true_file_name = file.name
            file.name = 'file'
        except KeyError:
            return Response(
                errors.format_message('file', errors.NO_FILES_GIVEN_IN_REQUEST),
                status=status.HTTP_400_BAD_REQUEST,
            )
        try:
            attachment = Attachment.objects.create(
                name=true_file_name,
                uploader=self.request.user,
                attached_file=file,
            )
        except APIError:
            return Response(
                errors.format_message('file', errors.UPLOAD_TO_MDS_FAILED),
                status=status.HTTP_400_BAD_REQUEST,
            )
        return Response({'id': attachment.id}, status=status.HTTP_200_OK)


class AttachmentView(BaseView):

    permission_classes = [
        ServicePermission('permissions.can_see_all_attachments'),
    ]

    def get(self, request, *args, **kwargs):
        """
        Скачивание файла
        """
        attachment = get_object_or_40x(Attachment, pk=self.kwargs['pk'])

        if request.yauser.authenticated_by.mechanism_name != 'tvm':
            # Для обратной совместимости. Выпилить, когда полностью перейдем на TVM от Docviewer
            if request.user.has_perm('permissions.can_see_all_attachments'):
                uid = request.query_params.get('uid')
                if uid:
                    user = User.objects.get(uid=uid)
                    context.init(user)

        if not is_attachment_available(attachment):
            raise PermissionDenied

        content_type = mimetypes.guess_type(attachment.name)[0]
        response = HttpResponse(
            attachment.attached_file,
            status=200,
            content_type=content_type or 'text/plain',
        )
        name = quote(attachment.name.encode('utf-8'))
        response['Content-Disposition'] = 'attachment; filename*=UTF-8\'\'{}'.format(name)
        return response
