from collections import OrderedDict

import sform
import waffle

from django.contrib.auth import get_user_model
from django.core.exceptions import ValidationError
from django.core.validators import validate_email

from intranet.femida.src.api.core.forms import (
    ApplicationChoiceField,
    AttachmentMultipleSuggestField,
    BaseMetaForm,
    BasePartialForm,
    CandidateMultipleSuggestField,
    CityMultipleSuggestField,
    CounterChoiceField,
    EMPTY_LABEL,
    InterviewMultipleChoiceField,
    MultipleSuggestField,
    ProfessionMultipleSuggestField,
    ProfessionSuggestField,
    SkillMultipleSuggestField,
    TagMultipleSuggestField,
    UserChoiceField,
    UserMultipleSuggestField,
    UserSuggestField,
    grid_values_matcher_by_id,
    validate_recruiters,
    PositiveIntegerField,
    ConditionallyRequiredFieldsMixin,
)
from intranet.femida.src.api.core.fields import NonStrictFieldsetField
from intranet.femida.src.api.core.validators import validate_phone, validate_yt_table
from intranet.femida.src.api.offers.forms import ChoiceField
from intranet.femida.src.applications.helpers import active_applications_query
from intranet.femida.src.candidates.bulk_upload.choices import CANDIDATE_UPLOAD_MODES
from intranet.femida.src.candidates.choices import (
    CANDIDATE_DEGREES,
    CANDIDATE_RESPONSIBLE_ROLES,
    CANDIDATE_SORTING_TYPES,
    CANDIDATE_STATUSES,
    CONSIDERATION_RECRUITER_STAGES,
    CONSIDERATION_RESOLUTIONS,
    CONTACT_TYPES,
    RKN_CONTACT_TYPES,
    CANDIDATE_COST_TYPES_MONEY,
    CANDIDATE_COST_RATES,
    VERIFICATION_TYPES,
)
from intranet.femida.src.candidates.considerations.controllers import get_relevant_consideration
from intranet.femida.src.candidates.contacts import validate_contact_url
from intranet.femida.src.candidates.controllers import (
    count_candidates_per_recruiter_stage,
    count_candidates_per_responsible_role,
    filter_candidates_by_responsible_role,
)
from intranet.femida.src.candidates.deduplication import DuplicatesFinder
from intranet.femida.src.candidates.models import (
    Candidate,
    CandidateContact,
    CandidateEducation,
    CandidateJob,
    CandidateProfession,
    ScoringCategory,
)
from intranet.femida.src.core.choices import GENDER_CHOICES
from intranet.femida.src.core.db.fields import StartrekIssueKeyField
from intranet.femida.src.core.models import Currency
from intranet.femida.src.interviews.choices import INTERVIEW_HIRE_GRADES, APPLICATION_RESOLUTIONS
from intranet.femida.src.interviews.models import Interview, Application
from intranet.femida.src.offers.choices import SOURCES
from intranet.femida.src.staff.choices import DEPARTMENT_TAGS
from intranet.femida.src.staff.models import Department
from intranet.femida.src.utils.switches import is_candidate_approval_with_application_enabled
from intranet.femida.src.utils.switches import is_candidate_main_recruiter_enabled
from intranet.femida.src.vacancies.choices import VACANCY_PRO_LEVELS, VACANCY_TYPES


User = get_user_model()


def _get_applications_queryset(candidate_id):
    # полагаемся на проверку прав на кандидате
    return Application.unsafe.filter(
        active_applications_query,
        consideration=get_relevant_consideration(candidate_id),
    )


class ResponsiblesFormMixin(sform.SForm):

    main_recruiter = UserSuggestField(state=sform.READONLY)
    recruiters = UserMultipleSuggestField()
    responsibles = UserMultipleSuggestField()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        if is_candidate_main_recruiter_enabled():
            self.fields.pop('responsibles', None)
        else:
            self.fields.pop('main_recruiter', None)
            self.fields.pop('recruiters', None)

    def clean_main_recruiter(self, main_recruiter):
        validate_recruiters([main_recruiter], allow_recruiter_assessor=True)
        return main_recruiter

    def clean_recruiters(self, recruiters):
        validate_recruiters(recruiters, allow_recruiter_assessor=True)
        return recruiters

    def clean_responsibles(self, responsibles):
        validate_recruiters(responsibles, allow_recruiter_assessor=True)
        return responsibles


class CandidateContactForm(sform.SForm):
    id = sform.IntegerField(state=sform.READONLY)
    type = sform.ChoiceField(state=sform.REQUIRED, choices=CONTACT_TYPES)
    account_id = sform.CharField(max_length=255)
    is_main = sform.BooleanField(default=False)

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        if waffle.switch_is_active('is_rkn'):
            self.fields['type'].choices = RKN_CONTACT_TYPES

    def clean_account_id(self, value):
        value = value.strip()
        if not value:
            return value

        contact_type = self.cleaned_data.get('type')

        if contact_type == CONTACT_TYPES.email:
            validate_email(value)
        elif contact_type == CONTACT_TYPES.phone:
            validate_phone(value)
        elif contact_type in (CONTACT_TYPES.hh, CONTACT_TYPES.ah, CONTACT_TYPES.linkedin):
            validate_contact_url(contact_type, value)
        return value


class CandidateEducationForm(sform.SForm):
    id = sform.IntegerField(state=sform.READONLY)
    institution = sform.CharField(state=sform.REQUIRED, max_length=255)
    faculty = sform.CharField(max_length=255)
    degree = sform.ChoiceField(state=sform.REQUIRED, choices=CANDIDATE_DEGREES)
    end_date = sform.DateField()


class CandidateJobForm(sform.SForm):
    id = sform.IntegerField(state=sform.READONLY)
    employer = sform.CharField(state=sform.REQUIRED, max_length=255)
    position = sform.CharField(max_length=255)
    start_date = sform.DateField(state=sform.REQUIRED)
    end_date = sform.DateField()
    salary_evaluation = sform.CharField(max_length=255)


class CandidateProfessionForm(sform.SForm):
    id = sform.IntegerField(state=sform.READONLY)
    profession = ProfessionSuggestField(state=sform.REQUIRED)
    salary_expectation = sform.CharField(max_length=255)


class CandidateCreateForm(ResponsiblesFormMixin, BasePartialForm):

    first_name = sform.CharField(max_length=255)
    middle_name = sform.CharField(max_length=255)
    last_name = sform.CharField(state=sform.REQUIRED, max_length=255)
    birthday = sform.DateField()
    gender = sform.ChoiceField(choices=GENDER_CHOICES)
    country = sform.CharField(max_length=255)
    city = sform.CharField(max_length=255)
    target_cities = CityMultipleSuggestField()
    contacts = sform.GridField(
        field_instance=sform.FieldsetField(CandidateContactForm),
        values_matcher=grid_values_matcher_by_id,
    )
    educations = sform.GridField(
        field_instance=sform.FieldsetField(CandidateEducationForm),
        values_matcher=grid_values_matcher_by_id,
    )
    jobs = sform.GridField(
        field_instance=sform.FieldsetField(CandidateJobForm),
        values_matcher=grid_values_matcher_by_id,
    )
    skills = SkillMultipleSuggestField()
    tags = TagMultipleSuggestField(create_missing=True)
    candidate_professions = sform.GridField(
        field_instance=sform.FieldsetField(CandidateProfessionForm),
        values_matcher=grid_values_matcher_by_id,
    )
    attachments = AttachmentMultipleSuggestField()
    source = ChoiceField(choices=SOURCES)
    source_description = sform.CharField(max_length=255)
    is_hidden = sform.BooleanField()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        if self.initial.get('status') == CANDIDATE_STATUSES.closed:
            consideration_fields = (
                'source',
                'source_description',
                'responsibles',
                'recruiters',
                'main_recruiter',
            )
            for field in consideration_fields:
                self.fields.pop(field, None)

        self.ignore_duplicates = self.context.get('ignore_duplicates')

    def _get_duplicates(self):
        finder = DuplicatesFinder(base_qs=Candidate.objects.all())
        return [
            {'id': duplicate.id, 'details': info.to_dict()}
            for duplicate, info, decision in finder.find_top3(self.cleaned_data)
        ]

    def clean_contacts(self, contacts):
        """
        Мы сделали поле account_id необязательным у контакта,
        поэтому тут мы вырезаем все контакты с пустым account_id,
        а также проверяем, что есть хотя бы один заполненный контакт
        https://st.yandex-team.ru/FEMIDA-1336
        """
        contacts = [
            contact
            for contact in contacts
            if contact['account_id']
        ]
        if not contacts:
            raise ValidationError(
                message='Please specify at least 1 contact',
                code='required'
            )
        return contacts

    def clean(self):
        if not self.ignore_duplicates:
            duplicates = self._get_duplicates()
            if duplicates:
                raise ValidationError(
                    message='Duplicates were found.',
                    code='duplicates_were_found',
                    params={'duplicates': duplicates},
                )

        return self.cleaned_data


class CandidateUpdateForm(CandidateCreateForm):

    main_recruiter = UserSuggestField(state=sform.REQUIRED)
    responsibles = UserMultipleSuggestField(state=sform.REQUIRED)


class HHCandidateForm(CandidateCreateForm):
    """
    Форма кандидата с доп.данными о файлах HH,
    доступных для прикрепления.
    """
    def meta_as_dict(self):
        return {
            'hh_attachments': self.initial.get('hh_attachments', []),
            'has_missing_fields': self.initial['has_missing_fields'],
        }

    def as_dict(self):
        result = super().as_dict()
        result['meta'] = self.meta_as_dict()
        return result


class CandidateMergeForm(sform.SForm):

    id = CandidateMultipleSuggestField()
    first_name = sform.CharField(max_length=255)
    middle_name = sform.CharField(max_length=255)
    last_name = sform.CharField(max_length=255)
    birthday = sform.DateField()
    gender = sform.ChoiceField(choices=GENDER_CHOICES)
    country = sform.CharField(max_length=255)
    city = sform.CharField(max_length=255)
    source = sform.ChoiceField(choices=SOURCES)
    source_description = sform.CharField(max_length=255)
    main_recruiter = UserSuggestField()
    inn = sform.BooleanField()

    contacts = MultipleSuggestField(queryset=CandidateContact.objects.all())
    educations = MultipleSuggestField(queryset=CandidateEducation.objects.all())
    jobs = MultipleSuggestField(queryset=CandidateJob.objects.all())
    candidate_professions = MultipleSuggestField(queryset=CandidateProfession.objects.all())
    skills = SkillMultipleSuggestField()
    attachments = AttachmentMultipleSuggestField()
    target_cities = CityMultipleSuggestField()
    tags = TagMultipleSuggestField()


class CandidateCompareForm(CandidateMergeForm):
    """
    Форма, которая формируем данные для сравнения нескольких кандидатов.
    Принимает на вход список сериализованных кандидатов, вместо одного
    value каждого поля оборачивается в options.
    Полям добавляется флаг conflict.
    Значениям options добавляются флаги checked.
    """
    login = sform.CharField(max_length=64)

    def default_getter(self, name):
        """
        itemgetter по каждому элементу списка
        """
        return lambda obj: [i[name] for i in obj]

    def as_compare_dict(self, similarity_info):
        result = OrderedDict()
        for name, field in self.fields.items():
            options = self._get_initial_value(name, field)
            state = self.get_field_state(name)
            result[name] = field._get_base_field_dict(
                prefix='',
                name=name,
                state=state,
            )
            result[name]['conflict'] = (
                name in similarity_info.conflict_fields
                or (
                    name == 'main_recruiter'
                    and self._has_different_main_recruiters(options)
                )
            )
            result[name]['options'] = []
            for i, option in enumerate(options):
                if isinstance(option, list):
                    result[name]['options'].append([
                        {'value': item, 'checked': True} for item in option
                    ])
                else:
                    result[name]['options'].append({
                        'value': option,
                        'checked': i == 0,  # для радио выбираем везде первый
                    })
            if name == 'inn':
                result[name]['options'] = []
                if similarity_info.match_details.get(name):
                    result[name]['options'].append({
                        'value': '',
                    })

        return result

    def _has_different_main_recruiters(self, data):
        return len({user['username'] for user in data if user}) > 1


class CandidatesCheckForDuplicatesForm(sform.SForm):
    """
    Форма для подготовки данных
    для поиска возможных дублей
    """
    first_name = sform.CharField(max_length=255)
    middle_name = sform.CharField(max_length=255)
    last_name = sform.CharField(max_length=255)
    attachments = AttachmentMultipleSuggestField()
    contacts = sform.GridField(
        field_instance=sform.FieldsetField(CandidateContactForm),
        values_matcher=grid_values_matcher_by_id,
    )

    def clean_contacts(self, contacts):
        return [c for c in contacts if c['account_id']]


class CandidateOpenForm(ResponsiblesFormMixin):

    responsibles = UserMultipleSuggestField(state=sform.REQUIRED)
    source = ChoiceField(SOURCES, state=sform.REQUIRED)
    source_description = sform.CharField(max_length=255)


class CandidateCloseForm(sform.SForm):

    close_for_recruiter = sform.NullBooleanField(state=sform.REQUIRED)
    # TODO: избавиться от этого поля после релиза FEMIDA-5725
    application_resolution = sform.ChoiceField(
        empty_label=EMPTY_LABEL,
        choices=CONSIDERATION_RESOLUTIONS,
        state=sform.REQUIRED,
    )
    consideration_resolution = sform.ChoiceField(
        empty_label=EMPTY_LABEL,
        choices=CONSIDERATION_RESOLUTIONS,
    )
    main_recruiter = UserChoiceField(queryset=User.objects.none())

    @property
    def candidate(self):
        return self.context['candidate']

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        if waffle.switch_is_active('enable_new_candidate_close_form'):
            self.fields.pop('application_resolution')
        else:
            self.fields.pop('consideration_resolution')
            self.fields.pop('close_for_recruiter')
            self.fields.pop('main_recruiter')

        # если у кандидата единственный ответственный, то
        # закрыть кандидата можно только целиком
        if not self.candidate.recruiters:
            self.fields.pop('close_for_recruiter', None)
            self.fields.pop('main_recruiter', None)

        if self.candidate.main_recruiter != self.context['user']:
            self.fields.pop('main_recruiter', None)

        if 'main_recruiter' in self.fields:
            self.fields['main_recruiter'].queryset = (
                User.objects.filter(
                    candidate_responsibles__candidate=self.candidate,
                    candidate_responsibles__role=CANDIDATE_RESPONSIBLE_ROLES.recruiter,
                )
                .order_by('username')
            )

    def get_field_state(self, name):
        if name == 'main_recruiter':
            must_set_new_main_recruiter = (
                self.cleaned_data.get('close_for_recruiter')
                and self.candidate.main_recruiter == self.context['user']
            )
            if must_set_new_main_recruiter:
                return sform.REQUIRED

        if name == 'consideration_resolution':
            close_candidate = self.cleaned_data.get('close_for_recruiter') is False
            if close_candidate:
                return sform.REQUIRED

        return super().get_field_state(name)

    def clean(self):
        cleaned_data = super().clean()
        application_resolution = cleaned_data.get('application_resolution')
        cleaned_data.setdefault('consideration_resolution', application_resolution)
        cleaned_data['application_resolution'] = APPLICATION_RESOLUTIONS.consideration_archived
        return cleaned_data


class CandidateSearchForm(sform.SForm):

    EMPTY_VALUES = (None, '')

    professions = ProfessionMultipleSuggestField()
    skills = SkillMultipleSuggestField()
    target_cities = CityMultipleSuggestField()
    tags = TagMultipleSuggestField()
    without_nohire = sform.BooleanField()
    skype_interviews_avg_grade = sform.ChoiceField(
        empty_label=EMPTY_LABEL,
        choices=INTERVIEW_HIRE_GRADES,
    )
    on_site_interviews_avg_grade = sform.ChoiceField(
        empty_label=EMPTY_LABEL,
        choices=INTERVIEW_HIRE_GRADES,
    )
    city = sform.CharField()
    institution = sform.CharField()
    employer = sform.CharField()
    responsibles = UserMultipleSuggestField()
    ignore_employees = sform.BooleanField()
    is_active = sform.NullBooleanField()

    created__gte = sform.DateField()
    created__lte = sform.DateField()
    modified__gte = sform.DateField()
    modified__lte = sform.DateField()

    sort = sform.ChoiceField(
        choices=CANDIDATE_SORTING_TYPES,
        default=CANDIDATE_SORTING_TYPES.relevance,
    )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        user = self.context.get('user')

        if not user or not user.is_recruiter:
            self.fields.pop('ignore_employees')

    def clean(self):
        self.cleaned_data = {
            k: v for k, v in self.cleaned_data.items() if v not in self.EMPTY_VALUES
        }
        return self.cleaned_data


class CandidateCreateProposalsForm(sform.SForm):

    cities = CityMultipleSuggestField(state=sform.REQUIRED)
    professions = ProfessionMultipleSuggestField(state=sform.REQUIRED)
    skills = SkillMultipleSuggestField(state=sform.REQUIRED)
    pro_level_min = sform.ChoiceField(
        empty_label=EMPTY_LABEL,
        choices=VACANCY_PRO_LEVELS,
    )
    pro_level_max = sform.ChoiceField(
        empty_label=EMPTY_LABEL,
        choices=VACANCY_PRO_LEVELS,
        state=sform.REQUIRED,
    )
    interviews = InterviewMultipleChoiceField(queryset=Interview.unsafe.none())
    departments = MultipleSuggestField(
        queryset=Department.objects.alive(),
    )
    comment = sform.CharField()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        interviews = self.context.get('interviews')
        assert interviews is not None
        self.fields['interviews'].queryset = interviews

    def _clean_pro_level(self, level):
        if level is None or level == '':
            return None
        try:
            return int(level)
        except ValueError:
            return level

    clean_pro_level_min = _clean_pro_level
    clean_pro_level_max = _clean_pro_level


class CandidateCreateProposalsFrontendForm(CandidateCreateProposalsForm):

    departments = sform.ModelMultipleChoiceField(
        queryset=Department.objects
            .alive()
            .filter(
                tags__overlap=[
                    DEPARTMENT_TAGS.business_unit,
                    DEPARTMENT_TAGS.experiment,
                ],
            ),
        label_extractor=lambda x: x.localized_name,
    )


class CandidateSendForApprovalForm(sform.SForm):

    application = ApplicationChoiceField(
        queryset=Application.unsafe.none(),
        empty_label=EMPTY_LABEL,
    )

    # TODO: выпилить после релиза FEMIDA-4813
    startrek_key = sform.CharField()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.candidate = self.context['candidate']
        self.fields['application'].queryset = (
            _get_applications_queryset(self.candidate.id)
            .exclude(vacancy__type=VACANCY_TYPES.pool)
        )

    def get_field_state(self, name):
        is_application_enabled = is_candidate_approval_with_application_enabled()
        required_fields = {
            'startrek_key': not is_application_enabled,
            'application': is_application_enabled,
        }
        if name in required_fields:
            return sform.REQUIRED if required_fields[name] else sform.NORMAL
        return super().get_field_state(name)

    def clean_startrek_key(self, value):
        if not value:
            return value
        return StartrekIssueKeyField().clean(value, self.candidate)


class CandidateCloseIrrelevantApplicationsForm(BaseMetaForm):

    def meta_as_dict(self):
        return {
            'irrelevant_applications_count': self.context['irrelevant_applications'].count(),
        }


class CandidateRecruiterListFilterForm(sform.SForm):

    responsible_role = CounterChoiceField(
        choices=CANDIDATE_RESPONSIBLE_ROLES,
        empty_label=EMPTY_LABEL,
    )
    stage = CounterChoiceField(
        choices=CONSIDERATION_RECRUITER_STAGES,
        empty_label=EMPTY_LABEL,
    )

    def structure_as_dict(self, *args, **kwargs):
        candidates = self.context['candidates']
        user = self.context['user']

        counts_by_responsible_role = count_candidates_per_responsible_role(candidates, user)
        self.fields['responsible_role'].counts = counts_by_responsible_role

        if self.initial.get('responsible_role'):
            candidates = filter_candidates_by_responsible_role(
                qs=candidates,
                user=user,
                role=self.initial['responsible_role'],
            )

        self.fields['stage'].counts = count_candidates_per_recruiter_stage(candidates)
        return super().structure_as_dict(*args, **kwargs)


class CandidateCreateVerificationForm(sform.SForm):
    type = sform.ChoiceField(
        choices=VERIFICATION_TYPES,
        state=sform.REQUIRED,
    )

    sender = sform.EmailField(state=sform.READONLY)
    receiver = sform.EmailField(state=sform.REQUIRED)
    subject = sform.CharField(state=sform.REQUIRED)
    text = sform.CharField(state=sform.REQUIRED)
    application = ApplicationChoiceField(
        queryset=Application.unsafe.none(),
        empty_label=EMPTY_LABEL,
        state=sform.REQUIRED,
    )
    uuid = sform.CharField(state=sform.REQUIRED)

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        candidate = self.context['candidate']
        self.fields['application'].queryset = _get_applications_queryset(candidate.id)


class CandidateCreateInternationalVerificationForm(
    ConditionallyRequiredFieldsMixin,
    CandidateCreateVerificationForm,
):
    type = sform.ChoiceField(
        choices=VERIFICATION_TYPES,
        state=sform.REQUIRED,
    )

    grade = PositiveIntegerField()

    CONDITIONALLY_REQUIRED = {
        'grade': ('type', VERIFICATION_TYPES.international_by_grade),
    }


class YTCandidateBulkUploadForm(sform.SForm):

    table = sform.CharField(
        state=sform.REQUIRED,
        validators=[validate_yt_table],
    )
    mode = sform.ChoiceField(
        choices=CANDIDATE_UPLOAD_MODES,
        state=sform.REQUIRED,
    )


class YTCandidateScoringBulkUploadForm(sform.SForm):

    table = sform.CharField(
        state=sform.REQUIRED,
        validators=[validate_yt_table],
    )
    version = sform.CharField(state=sform.REQUIRED, max_length=32)
    scoring_category = sform.ModelChoiceField(
        queryset=ScoringCategory.objects.all(),
        label_extractor=str,
        state=sform.REQUIRED,
    )


class CandidateCreateCostMoneyForm(sform.SForm):

    rate = sform.ChoiceField(
        choices=CANDIDATE_COST_RATES,
        state=sform.REQUIRED,
        default=CANDIDATE_COST_RATES.monthly,
    )
    value = sform.IntegerField(state=sform.REQUIRED)
    taxed = sform.BooleanField(state=sform.REQUIRED)
    currency = sform.ModelChoiceField(
        queryset=Currency.objects.filter(code__in=['RUB', 'USD', 'EUR']),
        label_extractor='code',
        state=sform.REQUIRED,
    )


class CandidateCreateCostShareForm(sform.SForm):

    value = sform.IntegerField(state=sform.REQUIRED)
    currency = sform.ModelChoiceField(
        queryset=Currency.objects.filter(code__in=['RUB', 'USD', 'EUR']),
        label_extractor='code',
        state=sform.NORMAL,
    )
    taxed = sform.BooleanField(state=sform.REQUIRED)


class CandidateCreateCostCommentForm(sform.SForm):

    comment = sform.CharField(state=sform.REQUIRED, max_length=255)


class CandidateCostMoneyForm(sform.SForm):

    rate = sform.ChoiceField(
        choices=CANDIDATE_COST_RATES,
        state=sform.REQUIRED,
        default=CANDIDATE_COST_RATES.monthly,
    )
    value = sform.IntegerField(state=sform.REQUIRED)
    taxed = sform.BooleanField(state=sform.REQUIRED)
    type = sform.ChoiceField(
        choices=CANDIDATE_COST_TYPES_MONEY,
        state=sform.REQUIRED,
        default=CANDIDATE_COST_TYPES_MONEY.total,
    )
    currency = sform.ModelChoiceField(
        queryset=Currency.objects.filter(code__in=['RUB', 'USD', 'EUR']),
        label_extractor='code',
        state=sform.REQUIRED,
    )


class CandidateCostShareForm(sform.SForm):

    value = sform.IntegerField(state=sform.REQUIRED)
    currency = sform.ModelChoiceField(
        queryset=Currency.objects.filter(code__in=['RUB', 'USD', 'EUR']),
        label_extractor='code',
        state=sform.NORMAL,
    )


class CandidateCostCommentForm(sform.SForm):

    comment = sform.CharField(max_length=255, state=sform.REQUIRED)


class CandidateCostsForm(sform.SForm):

    salary = NonStrictFieldsetField(CandidateCreateCostMoneyForm, state=sform.NORMAL, trim_on_empty=True)
    total = NonStrictFieldsetField(CandidateCreateCostMoneyForm, state=sform.NORMAL, trim_on_empty=True)
    salary_bonus = NonStrictFieldsetField(CandidateCreateCostMoneyForm, state=sform.NORMAL, trim_on_empty=True)
    rsu = NonStrictFieldsetField(CandidateCreateCostShareForm, state=sform.NORMAL, trim_on_empty=True)
    comment = NonStrictFieldsetField(CandidateCostCommentForm, state=sform.NORMAL, trim_on_empty=True)


class CandidateCostsSetForm(sform.SForm):

    expectation = NonStrictFieldsetField(CandidateCostsForm, state=sform.NORMAL)
    current = NonStrictFieldsetField(CandidateCostsForm, state=sform.NORMAL)
