import json
import waffle

from collections import defaultdict
from itertools import chain

from constance import config
from django.contrib.auth import get_user_model
from django.forms import BooleanField
from django.utils.functional import cached_property
from rest_framework import status
from rest_framework.exceptions import PermissionDenied
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response

from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.api.applications.serializers import CandidateApplicationSerializer
from intranet.femida.src.api.candidates import forms, serializers, permissions
from intranet.femida.src.api.considerations.serializers import InterviewRoundSerializer
from intranet.femida.src.api.core.errors import format_message
from intranet.femida.src.api.core.pagination import (
    ISearchPageNumberPagination,
    MagicLinksPagination,
    CandidateFilterPagination,
)
from intranet.femida.src.api.core.permissions import StrictServicePermission, IsRecruiter
from intranet.femida.src.api.core.views import (
    BaseView,
    BaseFormView,
    BaseFormViewMixin,
    InstanceFormViewMixin,
    QueryParamsFormView,
    WorkflowView,
    QueryParamValidatorSwitcherViewMixin,
    unfold_query_params,
    ResponseError,
    QueryParamsFormViewMixin,
    CursorPaginationCountListViewMixin,
)
from intranet.femida.src.api.hire_orders.permissions import ForbiddenForAutohire
from intranet.femida.src.api.interviews.forms import InterviewRoundCreateForm
from intranet.femida.src.api.tracker.views import TrackerTriggerBaseView
from intranet.femida.src.applications.helpers import (
    active_applications_query,
    get_candidate_visible_interviews,
    annotate_application_qs_for_serialization,
)
from intranet.femida.src.candidates import deduplication
from intranet.femida.src.candidates.amazing_hiring.serializers import (
    AmazingHiringCandidateSerializer,
)
from intranet.femida.src.candidates.choices import (
    CONSIDERATION_STATUSES,
    VERIFICATION_STATUSES,
    CANDIDATE_SORTING_TYPES,
    VERIFICATION_TYPES,
)
from intranet.femida.src.candidates.controllers import (
    get_candidates_extended_statuses,
    get_candidates_extended_statuses_and_changed_at,
    add_extended_status_order_to_candidates_qs,
    filter_candidates_by_recruiter_stage,
    filter_candidates_by_responsible_role,
    CandidateCostHistory,
    CandidateCostSetNotFound,
    CandidateCostsSetMustContainsAtLeastOneCost,
    UserHasNoAccessToCandidateCostsSet,
)
from intranet.femida.src.candidates.filters.base import FilterCtl
from intranet.femida.src.candidates.helpers import (
    get_actual_candidates,
    candidate_is_current_employee_subquery,
    get_candidate_ids_for_recruiter,
)
from intranet.femida.src.candidates.hh.serializers import HHCandidateSerializer
from intranet.femida.src.candidates.models import Candidate, DuplicationCase, Verification
from intranet.femida.src.candidates.tasks import (
    resolve_verification_task,
    yt_bulk_upload_candidates_task,
    yt_bulk_upload_candidate_scorings_task,
)
from intranet.femida.src.candidates.workflow import CandidateWorkflow
from intranet.femida.src.core.shortcuts import get_object_or_40x
from intranet.femida.src.hh.api import HeadhunterAPI
from intranet.femida.src.hh.exceptions import HeadhunterError
from intranet.femida.src.interviews.models import Application
from intranet.femida.src.isearch.exceptions import ISearchError
from intranet.femida.src.isearch.result_collections import CandidateSearchCollection
from intranet.femida.src.utils.amazing_hiring import AmazingHiringAPI, AmazingHiringError
from intranet.femida.src.vacancies.models import Vacancy, VacancyMembership


User = get_user_model()


def unfold_search_query_params(params):
    return unfold_query_params(
        query_params=params,
        list_fields=('professions', 'skills', 'responsibles', 'target_cities', 'tags'),
    )


def _get_candidate_maps(candidate_ids):
    applications_qs = (
        Application.unsafe
        .filter(
            active_applications_query,
            candidate__in=candidate_ids,
        )
    )
    applications = (
        applications_qs.values('id', 'status', 'resolution', 'vacancy_id', 'candidate_id')
    )
    vac_candidates_map = defaultdict(set)
    cand_appls_map = defaultdict(list)
    for appl in applications:
        candidate_id = appl.pop('candidate_id')
        vac_candidates_map[appl['vacancy_id']].add(candidate_id)
        cand_appls_map[candidate_id].append(appl)

    vacancies = (
        Vacancy.objects
        .filter(id__in=applications_qs.values('vacancy'))
        .values('id', 'name', 'status')
    )

    cand_vacancies_map = defaultdict(list)
    for vacancy in vacancies:
        for candidate_id in vac_candidates_map[vacancy['id']]:
            cand_vacancies_map[candidate_id].append(vacancy)

    return {
        'candidate_vacancies_map': cand_vacancies_map,
        'candidate_applications_map': cand_appls_map,
    }


class CandidatesSearchView(BaseView):

    pagination_class = ISearchPageNumberPagination

    def list(self, request, *args, **kwargs):
        text = request.query_params.get('text', '').strip()

        params = unfold_search_query_params(request.query_params)
        filter_form = forms.CandidateSearchForm(
            data=params,
            partial=True,
            context={'user': request.user},
        )
        self.validate(filter_form)

        filter_params = filter_form.cleaned_data
        if not filter_params and not text:
            return Response(
                data={'error': [{
                    'code': 'empty_search_query',
                    'message': '`text` param or any filter param are required',
                    'params': {},
                }]},
                status=status.HTTP_400_BAD_REQUEST,
            )

        search_result_collection = CandidateSearchCollection(
            text=text,
            user_ticket=self.user_ticket,
            ip=self.client_ip,
            filter_params=filter_params,
            user=request.user,
        )
        paginator = self.pagination_class()
        try:
            data = paginator.paginate_queryset(
                queryset=search_result_collection,
                request=request,
                view=self,
            )
        except ISearchError:
            return Response(
                data={'detail': (
                    'Поиск не отвечает. Повторите, пожалуйста, запрос. '
                    'Если проблема останется, напишите на @tools.'
                )},
                status=status.HTTP_500_INTERNAL_SERVER_ERROR,
            )
        response = paginator.get_paginated_response(data)
        response.data['meta'] = {
            'isearch_url': search_result_collection._request_url,
        }
        return response

    def get(self, request, *args, **kwargs):
        """
        Поиск по кандидатам
        """
        return self.list(request, *args, **kwargs)


class CandidatesSearchFormView(QueryParamsFormView):

    model_class = Candidate
    validator_class = forms.CandidateSearchForm
    form_serializer_class = serializers.CandidateSearchFormSerializer

    def get_query_params(self):
        return unfold_search_query_params(self.request.query_params)


class CandidateViewMixin:

    def get_validator_context(self):
        return {
            'ignore_duplicates': self.request.data.get('ignore_duplicates'),
        }


class CandidateListCreateView(CandidateViewMixin, BaseView):

    model_class = Candidate
    list_item_serializer_class = serializers.CandidateLiteSerializer
    detail_serializer_class = serializers.CandidateSerializer
    validator_class = forms.CandidateCreateForm

    def get_queryset(self):
        return self.model_class.objects.alive().order_by('-id')

    def get(self, request, *args, **kwargs):
        """
        Создание кандидата
        """
        return self.list(request, *args, **kwargs)

    def filter_queryset(self, queryset):
        params = self.request.query_params
        self.only_related_to_user = BooleanField().to_python(params.get('only_related_to_user'))
        only_actual = BooleanField().to_python(params.get('only_actual'))
        order_by = ('-modified',)

        for_user = params.get('for')
        for_user = User.objects.filter(username=for_user).first() if for_user else self.request.user
        if not for_user:
            return queryset.none()

        if only_actual:
            queryset = get_actual_candidates(for_user)

        if self.only_related_to_user:
            queryset = queryset.filter(
                id__in=get_candidate_ids_for_recruiter(for_user),
                considerations__state=CONSIDERATION_STATUSES.in_progress,
            )
            queryset = add_extended_status_order_to_candidates_qs(queryset)
            order_by = ('extended_status_order', '-modified')

        queryset = (
            queryset
            .annotate(is_current_employee=candidate_is_current_employee_subquery)
            .order_by(*order_by)
        )
        return queryset

    def get_list_item_serializer_context(self):
        candidate_ids = [c.id for c in self.page]
        context = _get_candidate_maps(candidate_ids)
        if not self.only_related_to_user:
            context['extended_statuses'] = get_candidates_extended_statuses(candidate_ids)
        return context

    @action_logged('candidate_create')
    def post(self, request, *args, **kwargs):
        """
        Создание кандидата
        """
        return self.create(request, *args, **kwargs)

    def perform_create(self, data):
        workflow = CandidateWorkflow(instance=None, user=self.request.user)
        create_action = workflow.get_action('create')
        if not create_action.is_available():
            raise PermissionDenied('workflow_error')
        return create_action.perform(**data)


class CandidateRecruiterListView(QueryParamsFormViewMixin, BaseView):

    model_class = Candidate
    list_item_serializer_class = serializers.CandidateRecruiterListSerializer
    validator_class = forms.CandidateRecruiterListFilterForm
    form_serializer_class = serializers.CandidateRecruiterListFilterFormSerializer

    def get(self, request, *args, **kwargs):
        """
        Мои кандидаты на главной
        """
        return self.list(request, *args, **kwargs)

    @cached_property
    def for_user(self):
        params = self.request.query_params
        for_user = params.get('for')
        for_user = User.objects.filter(username=for_user).first() if for_user else self.request.user
        return for_user

    @cached_property
    def _queryset(self):
        if not self.for_user:
            return Candidate.unsafe.none()

        return (
            self.model_class.objects
            .alive()
            .filter(
                id__in=get_candidate_ids_for_recruiter(self.for_user),
                considerations__state=CONSIDERATION_STATUSES.in_progress,
            )
        )

    def get_queryset(self):
        return self._queryset

    def filter_queryset(self, queryset):
        queryset = add_extended_status_order_to_candidates_qs(queryset)

        filter_form = self.get_validator_object(self.request.query_params)
        self.validate(filter_form)
        filter_params = filter_form.cleaned_data

        if filter_params['responsible_role']:
            queryset = filter_candidates_by_responsible_role(
                qs=queryset,
                user=self.for_user,
                role=filter_params['responsible_role'],
            )

        if filter_params['stage']:
            queryset = filter_candidates_by_recruiter_stage(queryset, filter_params['stage'])

        return (
            queryset
            .annotate(is_current_employee=candidate_is_current_employee_subquery)
            .order_by('-extended_status_order', '-modified')
        )

    def get_list_item_serializer_context(self):
        candidate_ids = [c.id for c in self.page]
        return _get_candidate_maps(candidate_ids)

    def get_validator_context(self):
        return {
            'candidates': self.get_queryset(),
            'user': self.for_user,
        }


class CandidateDetailView(CandidateViewMixin, InstanceFormViewMixin, BaseView):

    model_class = Candidate
    detail_serializer_class = serializers.CandidateSerializer
    validator_class = forms.CandidateUpdateForm
    form_serializer_class = serializers.CandidateFormSerializer

    def get_queryset(self):
        return (
            self.model_class.objects
            .annotate(
                is_current_employee=candidate_is_current_employee_subquery,
            )
        )

    def get(self, request, *args, **kwargs):
        """
        Один кандидат
        """
        return self.retrieve(request, *args, **kwargs)

    @action_logged('candidate_update')
    def patch(self, request, *args, **kwargs):
        """
        Частичное редактирование кандидата
        """
        return self.partial_update(request, *args, **kwargs)

    @action_logged('candidate_update')
    def put(self, request, *args, **kwargs):
        """
        Редактирование кандидата
        """
        return self.update(request, *args, **kwargs)

    def perform_update(self, data, instance):
        workflow = CandidateWorkflow(instance=instance, user=self.request.user)
        update_action = workflow.get_action('update')
        if not update_action.is_available():
            raise PermissionDenied('workflow_error')
        return update_action.perform(**data)


class CandidateCreateFormView(BaseFormView):

    model_class = Candidate

    @property
    def hh_id(self):
        return self.request.query_params.get('hh_id')

    @property
    def amazing_hiring_id(self):
        return self.request.query_params.get('ah_id')

    def get_validator_class(self):
        return forms.HHCandidateForm if self.hh_id else forms.CandidateCreateForm

    def get_initial_data(self):
        initial_data = {'main_recruiter': self.request.user.username}
        if self.hh_id and self.amazing_hiring_id:
            raise ResponseError(
                data={'error': [{'code': 'multiple_sources_error'}]},
                status=status.HTTP_400_BAD_REQUEST,
            )

        if self.hh_id:
            hh = HeadhunterAPI(self.request.user)
            try:
                data = hh.get_full_resume(self.hh_id)
            except HeadhunterError as exc:
                raise ResponseError(
                    data={'error': [{'code': exc.message}]},
                    status=status.HTTP_400_BAD_REQUEST,
                )
            initial_data.update(HHCandidateSerializer(data).data)

        if self.amazing_hiring_id:
            try:
                data = AmazingHiringAPI.get_profile(self.amazing_hiring_id)
            except AmazingHiringError as exc:
                raise ResponseError(
                    data={'error': [{'code': exc.message}]},
                    status=status.HTTP_400_BAD_REQUEST,
                )
            initial_data.update(AmazingHiringCandidateSerializer(data).data)

        return initial_data


class CandidateCheckForDuplicatesView(BaseView):

    list_item_serializer_class = serializers.CandidateDuplicateWithDetailsSerializer
    validator_class = forms.CandidatesCheckForDuplicatesForm

    def list(self, request, *args, **kwargs):
        validator = self.get_validator_object(request.data)
        self.validate(validator)

        finder = deduplication.DuplicatesFinder(base_qs=Candidate.objects.all())
        duplicates = [
            {
                'candidate': candidate,
                'details': info.to_dict(),
            }
            for candidate, info, decision
            in finder.find_top3(validator.cleaned_data)
        ]
        serializer = serializers.CandidateDuplicateWithDetailsSerializer(duplicates, many=True)
        return Response(serializer.data)

    def post(self, request, *args, **kwargs):
        """
        Список возможных дублей кандидата
        """
        return self.list(request, *args, **kwargs)


class CandidateCompareView(BaseFormView):

    model_class = Candidate
    form_serializer_class = serializers.CandidateCompareFormSerializer
    validator_class = forms.CandidateCompareForm

    def get(self, request, *args, **kwargs):
        """
        Данные для формы сравнения кандидатов
        """
        def get_fields_for_prefetch(prefix=''):
            fields_for_prefetch = (
                'candidate_attachments__attachment',
                'candidate_skills__skill',
                'candidate_professions__profession__professional_sphere',
                'candidate_responsibles__user',
                'candidate_tags__tag',
                'contacts',
                'educations',
                'jobs',
            )
            return [prefix + f for f in fields_for_prefetch]

        duplication_case_id = request.query_params.get('duplication_case')

        cand_ids = request.query_params.get('candidates')
        if cand_ids:
            cand_ids = cand_ids.split(',')

        if duplication_case_id:
            duplication_case = get_object_or_40x(
                DuplicationCase.objects.prefetch_related(*chain(
                    get_fields_for_prefetch('first_candidate__'),
                    get_fields_for_prefetch('second_candidate__'),
                )),
                pk=duplication_case_id,
            )
            candidates = [duplication_case.first_candidate, duplication_case.second_candidate]
        else:
            # Убрал .alive(), потому что страница сравнения
            # для закрытого DuplicationCase ломалась
            candidates = (
                Candidate.objects
                .filter(pk__in=cand_ids)
                .prefetch_related(*get_fields_for_prefetch())
            )
            if len(candidates) < 2:
                # Если по правам пользователя обрезали до <2 кандидатов,
                # не удастся построить страницу сравнения
                raise PermissionDenied

        initial_data = [self.get_form_serializer_object(instance).data for instance in candidates]
        form = self.get_validator_object(initial=initial_data)
        return Response(form.as_compare_dict(
            deduplication.SimilarityInfo(candidates[0], candidates[1])
        ))


class CandidateMergeView(BaseView):

    model_class = Candidate
    detail_serializer_class = serializers.CandidateSerializer
    validator_class = forms.CandidateMergeForm

    def update(self, request, *args, **kwargs):
        raw_data_keys = set(request.data.keys())
        validator = self.get_validator_object(request.data)
        self.validate(validator)
        data = {}
        for key, value in validator.cleaned_data.items():
            if key in raw_data_keys:
                data[key] = value
        candidates = data.pop('id')
        merger = deduplication.CandidateMerger(candidates=candidates, extra_fields=data)
        new_candidate = merger.merge()
        serializer = self.get_detail_serializer_object(new_candidate)
        return Response(serializer.data, status=status.HTTP_200_OK)

    @action_logged('candidate_merge')
    def post(self, request, *args, **kwargs):
        """
        Объединение кандидатов
        """
        return self.update(request, *args, **kwargs)


class CandidateApplicationListView(BaseView):

    model_class = Application
    list_item_serializer_class = CandidateApplicationSerializer

    @cached_property
    def candidate(self):
        return get_object_or_40x(Candidate, pk=self.kwargs['pk'])

    @cached_property
    def is_user_responsible_for_candidate(self):
        return self.candidate.responsibles.filter(id=self.request.user.id).exists()

    def get_queryset(self):
        return (
            self.candidate.applications
            .filter(consideration__state=CONSIDERATION_STATUSES.in_progress)
            .filter(active_applications_query)
            .select_related(
                'vacancy__department',
                'submission__form',
                'submission__reference',
                'submission__rotation',
                'submission__publication',
                'created_by',
            )
            .order_by('-modified')
        )

    def filter_queryset(self, queryset):
        queryset = annotate_application_qs_for_serialization(queryset)

        if not self.is_user_responsible_for_candidate:
            queryset = queryset.filter(vacancy__in=(
                VacancyMembership.unsafe
                .filter(member=self.request.user)
                .values('vacancy')
            ))

        return queryset

    def get(self, request, *args, **kwargs):
        """
        Список претенденств
        """
        return self.list(request, *args, **kwargs)

    def get_list_item_serializer_context(self):
        vacancy_ids = [i.vacancy_id for i in self.page]

        if not self.is_user_responsible_for_candidate:
            available_vacancy_ids = set(vacancy_ids)
        else:
            available_vacancy_ids = set(
                Vacancy.objects
                .filter(id__in=vacancy_ids)
                .values_list('id', flat=True)
            )

        return {
            'available_vacancy_ids': available_vacancy_ids,
        }


class CandidateFilterView(CursorPaginationCountListViewMixin, BaseFormViewMixin, BaseView):

    model_class = Candidate
    validator_class = FilterCtl
    pagination_class = CandidateFilterPagination

    def get_queryset(self):
        return (
            self.model_class.objects
            .alive()
            .annotate(
                is_current_employee=candidate_is_current_employee_subquery,
            )
        )

    def filter_queryset(self, queryset):
        filter_form = self.get_validator_object(data=self.get_query_params())
        self.validate(filter_form)
        annotations = filter_form.get_annotations()
        query = filter_form.get_query()
        return queryset.annotate(**annotations).filter(query)

    def get_ordering(self):
        sort = self.get_query_params().get('sort')
        filters = self.get_query_params().get('filters')

        if any(item['field'] == 'CandidateScoringFilter' for item in filters):
            # При выборе фильтра CandidateScoringFilter queryset аннотируется
            # соответствующим scoring_value
            return '-scoring_value'

        if sort == CANDIDATE_SORTING_TYPES.modified_asc:
            return 'modified'
        elif sort == CANDIDATE_SORTING_TYPES.modified_desc:
            return '-modified'
        return '-id'

    def get_query_params(self):
        filters = self.request.query_params.get('filters')
        query_params = {}
        if filters:
            try:
                filters = json.loads(filters)
            except ValueError:
                raise ResponseError(
                    data=format_message('filters', 'invalid_json'),
                    status=status.HTTP_400_BAD_REQUEST,
                )
            query_params['filters'] = filters
        sort = self.request.query_params.get('sort')
        if sort:
            query_params['sort'] = sort
        return query_params

    def get_initial_data(self):
        initial_data = self.get_query_params()
        form = self.get_validator_object(data=initial_data)
        if form.is_valid():
            return initial_data
        return None

    def is_count_required_for_list(self):
        return not waffle.switch_is_active('disable_candidate_filter_list_count')

    def count(self, request, *args, **kwargs):
        return Response({'count': self.get_count()})

    def get_list_item_serializer_class(self):
        user = self.request.user
        if user.is_recruiter or user.is_recruiter_assessor:
            return serializers.RecruiterCandidateFilterSerializer
        return serializers.CandidateFilterSerializer

    def get_list_item_serializer_context(self):
        candidate_ids = [c.id for c in self.page]
        return {'extended_statuses': get_candidates_extended_statuses_and_changed_at(candidate_ids)}


class CandidateWorkflowView(WorkflowView):

    model_class = Candidate
    detail_serializer_class = serializers.CandidateSerializer
    workflow_class = CandidateWorkflow

    @property
    def actionlog_name(self):
        return 'candidate_%s' % self.action_name


class CandidateOpenView(BaseFormViewMixin, CandidateWorkflowView):

    action_name = 'open'
    validator_class = forms.CandidateOpenForm

    def get_initial_data(self):
        return {'main_recruiter': self.request.user.username}


class CandidateCloseView(BaseFormViewMixin, CandidateWorkflowView):

    validator_class = forms.CandidateCloseForm
    permission_classes = [
        permissions.NoDuplicatesPermission,
        ForbiddenForAutohire,
    ]

    @property
    def action_name(self):
        """
        close_for_recruiter - закрывает претенденства кандидата по вакансиям рекрутера
        close - полностью завершает работу с кандидатом
        """
        close_for_recruiter = BooleanField().to_python(self.request.data.get('close_for_recruiter'))
        return 'close_for_recruiter' if close_for_recruiter else 'close'

    def get_initial_data(self):
        # Получаем объект, чтобы запустить проверку прав даже в ручке формы
        self.get_object()
        return {}

    def get_validator_context(self):
        return {'candidate': self.get_object()}


class CandidateCreateProposalsBaseView(InstanceFormViewMixin, CandidateWorkflowView):
    action_name = 'create_proposals'
    form_serializer_class = serializers.CandidateCreateProposalsFormSerializer

    @cached_property
    def interviews(self):
        return get_candidate_visible_interviews(self.get_object())

    def get_validator_context(self):
        return {
            'interviews': self.interviews,
        }

    def get_additional_response_data(self):
        return self.action.extra_data


class CandidateCreateProposalsView(CandidateCreateProposalsBaseView):
    validator_class = forms.CandidateCreateProposalsForm


class CandidateCreateProposalsFormView(CandidateCreateProposalsBaseView):
    validator_class = forms.CandidateCreateProposalsFrontendForm


class CandidateSendForApprovalView(BaseFormViewMixin, CandidateWorkflowView):

    validator_class = forms.CandidateSendForApprovalForm
    action_name = 'send_for_approval'

    def get_wf_context_data(self):
        return {
            'session_id': self.session_id,
        }

    def get_validator_context(self):
        return {
            'candidate': self.get_object(),
        }


class CandidateCloseIrrelevantApplicationsView(BaseFormViewMixin, CandidateWorkflowView):

    validator_class = forms.CandidateCloseIrrelevantApplicationsForm
    action_name = 'close_irrelevant_applications'

    def get_validator_context(self):
        return {
            'irrelevant_applications': self.get_action(self.action_name).irrelevant_applications,
        }


class CandidateCreateVerificationView(
    QueryParamValidatorSwitcherViewMixin,
    InstanceFormViewMixin,
    CandidateWorkflowView,
):
    """
    Создание Verification и отправка кандидату анкеты КИ
    """
    action_name = 'create_verification'
    validator_class = forms.CandidateCreateInternationalVerificationForm
    form_serializer_class = serializers.CandidateCreateVerificationFormSerializer
    detail_serializer_class = serializers.VerificationDetailSerializer
    permission_classes = [
        permissions.NoDuplicatesPermission,
        ForbiddenForAutohire,
    ]

    validator_classes_map = {
        VERIFICATION_TYPES.default: forms.CandidateCreateVerificationForm,
        VERIFICATION_TYPES.international_by_grade: forms.CandidateCreateInternationalVerificationForm,
    }

    def get_validator_context(self):
        return {
            'candidate': self.get_object(),
        }

    def get_form_serializer_context(self):
        return self.request.query_params.dict()


class VerificationResolutionView(TrackerTriggerBaseView):
    """
    В тикете ESS вынесен окончательный вердикт
    по выставлению оффера кандидату
    """
    model_class = Verification
    issue_key_field = 'startrek_ess_key'
    task = resolve_verification_task
    valid_statuses = (
        VERIFICATION_STATUSES.on_ess_check,
    )


class YTCandidateBulkUploadView(BaseView):
    """
    Вьюха для запуска обработки списков кандидатов в YT
    """
    validator_class = forms.YTCandidateBulkUploadForm
    permission_classes = [permissions.YTCandidateUploadPermission]

    def post(self, request, *args, **kwargs):
        return self.create(request, *args, **kwargs)

    def perform_create(self, data):
        upload_sources = json.loads(config.YT_CANDIDATE_UPLOAD_SOURCES)
        source = upload_sources.get(self.request.user.username)
        yt_bulk_upload_candidates_task.delay(source=source, **data)


class YTCandidateScoringBulkUploadView(BaseView):
    """
    Вьюха для заливки скорингов кандидатов из YT
    """
    validator_class = forms.YTCandidateScoringBulkUploadForm
    permission_classes = [permissions.YTCandidateScoringPermission]

    def post(self, request, *args, **kwargs):
        return self.create(request, *args, **kwargs)

    def perform_create(self, data):
        data['scoring_category'] = data['scoring_category'].id
        yt_bulk_upload_candidate_scorings_task.delay(**data)


class MagicLinksCandidateListView(BaseView):

    model_class = Candidate
    list_item_serializer_class = serializers.CandidateLiteSerializer
    pagination_class = MagicLinksPagination
    permission_classes = [
        StrictServicePermission('permissions.can_access_magiclinks_data'),
    ]

    def filter_queryset(self, queryset):
        candidate_ids = self.request.query_params.get('id')
        candidate_ids = candidate_ids.split(',') if candidate_ids else []
        return queryset.filter(id__in=candidate_ids)

    def get(self, request, *args, **kwargs):
        """
        Ручка списка кандидатов для маг.ссылок
        """
        return self.list(request, *args, **kwargs)


class InterviewRoundCreateView(BaseFormViewMixin, CandidateWorkflowView):

    action_name = 'interview_round_create'
    validator_class = InterviewRoundCreateForm
    detail_serializer_class = InterviewRoundSerializer

    permission_classes = [
        IsAuthenticated,
        permissions.NoDuplicatesPermission,
        ForbiddenForAutohire,
    ]

    def get_initial_data(self):
        return {
            '_active_applications_qs': (
                Application.unsafe
                .filter(active_applications_query)
                .filter(candidate_id=self.get_object().id)
            ),
            '_candidate': self.get_object(),
            '_user': self.request.user,
        }

    def get_validator_object(self, *args, **kwargs):
        kwargs.setdefault('base_initial', self.get_initial_data())
        return super().get_validator_object(*args, **kwargs)


class CandidateCostsView(BaseFormViewMixin, BaseView):

    model_class = Candidate
    permission_classes = [IsRecruiter]
    validator_class = forms.CandidateCostsSetForm
    detail_serializer_class = serializers.CandidateAllCostsSerializer

    def get(self, request, *args, **kwargs):
        return self.retrieve(request, *args, **kwargs)

    @action_logged('candidatecost_create')
    def post(self, request, *args, **kwargs):
        return self.create(request, *args, **kwargs)

    @action_logged('candidatecost_update')
    def put(self, request, *args, **kwargs):
        return self.update(request, *args, **kwargs)

    def perform_create(self, data):
        candidate = self.get_object()
        user = self.request.user
        history = CandidateCostHistory(candidate)
        try:
            history.update_from_form(user, data)
        except CandidateCostsSetMustContainsAtLeastOneCost:
            raise ResponseError(data='candidate.costs.empty', status=status.HTTP_400_BAD_REQUEST)

    def perform_update(self, data, instance):
        user = self.request.user
        history = CandidateCostHistory(instance)
        try:
            history.edit_from_form(user, data)
        except CandidateCostsSetMustContainsAtLeastOneCost:
            raise ResponseError(data='candidate.costs.empty', status=status.HTTP_400_BAD_REQUEST)
        except CandidateCostSetNotFound:
            raise ResponseError(status=status.HTTP_404_NOT_FOUND)
        except UserHasNoAccessToCandidateCostsSet:
            raise ResponseError(status=status.HTTP_403_FORBIDDEN)
