from rest_framework import status
from rest_framework.response import Response

from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.api.categories.permissions import CategorySubscribePermission
from intranet.femida.src.api.core.views import BaseView
from intranet.femida.src.core.db.helpers import get_count_subquery
from intranet.femida.src.problems.controllers import (
    subscribe_to_category,
    unsubscribe_from_category,
)
from intranet.femida.src.problems.models import Category, Problem, Complaint

from . import serializers


class CategoryListView(BaseView):

    model_class = Category
    list_item_serializer_class = serializers.CategorySerializer

    def get(self, request, *args, **kwargs):
        """
        Список категорий
        """
        return self.list(request, *args, **kwargs)

    def get_list_item_serializer_context(self):
        user_category_ids = set(
            Category.objects
            .filter(subscribers=self.request.user)
            .values_list('id', flat=True)
        )
        return {
            'user_category_ids': user_category_ids,
        }

    def get_queryset(self):
        problems_count_sq = get_count_subquery(
            queryset=Problem.alive_objects.all(),
            reverse_related_name='categories',
        )
        complaints_count_sq = get_count_subquery(
            queryset=Complaint.objects.filter(is_active=True),
            reverse_related_name='problem__categories',
        )
        return (
            self.model_class.objects
            .annotate(
                problems_count=problems_count_sq,
                active_complaints_count=complaints_count_sq,
            )
        )


class CategorySubscribeView(BaseView):

    model_class = Category
    permission_classes = [CategorySubscribePermission]

    @action_logged('category_subscribe')
    def post(self, request, *args, **kwargs):
        category = self.get_object()
        subscribe_to_category(
            category=category,
            user=self.request.user,
        )
        return Response(data={}, status=status.HTTP_200_OK)


class CategoryUnsubscribeView(BaseView):

    model_class = Category
    permission_classes = [CategorySubscribePermission]

    @action_logged('category_unsubscribe')
    def post(self, request, *args, **kwargs):
        category = self.get_object()
        unsubscribe_from_category(
            category=category,
            user=self.request.user,
        )
        return Response(data={}, status=status.HTTP_200_OK)
